<?php

namespace Inside\CSV\Importers;

use Exception;
use Illuminate\Support\Arr;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;
use League\Csv\Reader;

class CSVImporter implements ImporterInterface
{
    /** @var array */
    protected $presenters;

    public function getEntries(): array
    {
        return config('csv.files');
    }

    public function prepareImport(): bool
    {
        $this->presenters = config('csv.presenters');

        return true;
    }

    /**
     * @param array $user
     * @param array $entry
     * @param BridgeContent $bridge
     * @return array
     */
    public function getSyncAttributes($user, array $entry, BridgeContent $bridge): array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        $data = [];
        foreach ($entry['fields'] as $modelField => $fileField) {
            if (is_object($fileField) && ($fileField instanceof \Closure)) {
                $data[$modelField] = $fileField($user, $bridge);
            } else {
                $data[$modelField] = $user[$fileField];
            }
            if (isset($this->presenters[$modelField]) && is_callable($this->presenters[$modelField])) {
                $data[$modelField] = $this->presenters[$modelField]($data[$modelField]);
            }
        }
        return $data;
    }

    /**
     * @param array $file
     * @param array|null $filter
     * @param array|null $test
     * @return array
     * @throws \League\Csv\Exception
     */
    public function getUsers($file, $filter, $test = null, int $limit = 0): array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        if ($test) {
            return [$test];
        }

        $csv = Reader::createFromPath($file['path'], 'r');

        $encoding = mb_detect_encoding($csv->getContent(), $this->getEncodings(), true);

        if (! $encoding) {
            throw new Exception('Encoding must be a valid UTF-8');
        }

        if ($encoding !== 'UTF-8') {
            $utf8_csv = mb_convert_encoding($csv->getContent(), 'UTF-8', $encoding);
            $csv = Reader::createFromString($utf8_csv);
        }

        if (mb_detect_encoding($csv->getContent(), $this->getEncodings(), true) !== 'UTF-8') {
            throw new Exception('Encoding must be a valid UTF-8');
        }

        $csv->setDelimiter($file['separator']);
        $csv->setHeaderOffset(0);
        if ($csv->count() === 0) {
            return [];
        }

        $headers = array_map(
            function ($item) {
                return trim(strtolower($item));
            },
            $csv->getHeader()
        );

        $users = iterator_to_array($csv->getRecords($headers));

        $postFilter = config('csv.postfilter');
        if ($postFilter && is_callable($postFilter)) {
            $users = $postFilter(collect($users));
        }

        return $users;
    }

    /**
     * Get the function cleanData (launched after retrieving data for a given user) from config file
     * Signature must be : function(&$data)
     *
     * @return mixed
     */
    public function getCleanData()
    {
        return config('csv.cleandata');
    }

    /**
     * Get the function preFlight (launched before the import) from the config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPreFlight()
    {
        return config('csv.preflight');
    }

    /**
     * Get the function postFlight (launched after the import) from config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPostFlight()
    {
        return config('csv.postflight');
    }

    public function getFilter(): ?string
    {
        return config('csv.filter', null);
    }

    /**
     * Get User information from AD, null if user not found.
     *
     * @param string|null $username
     * @param array $entry
     * @return array|null
     */
    public function getUserInformation($username, $entry): ?array
    {
        $users = $this->getUsers($entry, null);
        if (is_null($username)) {
            return Arr::first($users);
        }

        foreach ($users as $user) {
            if (in_array($username, $user)) {
                return $user;
            }
        }

        return null;
    }

    protected function getEncodings(): array
    {
        return [
            'BASE64',
            'UTF-32',
            'UTF-32BE',
            'UTF-32LE',
            'UTF-16',
            'UTF-16BE',
            'UTF-16LE',
            'UTF-8',
            'UTF-7',
            'UTF7-IMAP',
            'ASCII',
            'ISO-8859-1',
            'ISO-8859-2',
            'ISO-8859-3',
            'ISO-8859-4',
            'ISO-8859-5',
            'ISO-8859-6',
            'ISO-8859-7',
            'ISO-8859-8',
            'ISO-8859-9',
            'ISO-8859-10',
            'ISO-8859-13',
            'ISO-8859-14',
            'ISO-8859-15',
            'ISO-8859-16'
        ];
    }
}
