<?php

namespace Inside\CSV\Actions\FeedValidation;

use Illuminate\Support\Facades\Lang;
use Inside\CSV\Services\Import\ImportStrategy;
use Inside\Validation\ValidateRequests;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\HeadingRowImport;

class XlsxFeedValidationStrategy implements FeedValidationStrategy
{
    use ValidateRequests;

    private const VALID_EXTENSION = 'xlsx';

    public function validate(string $file, array $expectedColumns, callable $progressTrackerCallback): void
    {
        $this->assertValidFileType($progressTrackerCallback, $file);
        $this->assertColumnCount($progressTrackerCallback, $file, $expectedColumns);
        $this->assertColumnNamesMatch($progressTrackerCallback, $file, $expectedColumns);
        $this->assertValidRows($progressTrackerCallback, $file);
    }

    private function assertValidFileType(callable $progressTrackerCallback, string $file): void
    {
        $extension = pathinfo($file, PATHINFO_EXTENSION);
        if ($extension !== self::VALID_EXTENSION) {
            Lang::get('validation.invalid_file_type', [
                'type' => $extension,
                'expected' => self::VALID_EXTENSION
            ]);
        }
        $progressTrackerCallback([]);
    }

    private function assertColumnCount(callable $progressTrackerCallback, string $file, array $expectedColumns): void
    {
        $errors = [];
        $headings = (new HeadingRowImport())->toArray($file)[0][0] ?? [];
        if (count($headings) < count($expectedColumns)) {
            $errors[] = Lang::get('validation.invalid_column_count');
        }
        $progressTrackerCallback($errors);
    }

    private function assertColumnNamesMatch(callable $progressTrackerCallback, string $file, array $expectedColumns): void
    {
        $errors = [];
        $headings = (new HeadingRowImport())->toArray($file)[0][0] ?? [];
        $headings = array_map('trim', $headings);
        if (
            array_diff(
                array_map('mb_strtolower', $expectedColumns),
                array_map('mb_strtolower', $headings)
            )
        ) {
            $errors[] = Lang::get('validation.invalid_column_names');
        }
        $progressTrackerCallback($errors);
    }

    private function assertValidRows(callable $progressTrackerCallback, string $file): void
    {
        $rows = Excel::toArray((object)[], $file)[0] ?? [];
        $code = config('app.code');
        $emailFieldLabel = config('xlsx.files.' . $code.'.fields.mail') ?? config('xlsx.files.' . $code.'.fields.email');
        $nameFieldLabel = config('xlsx.files.' . $code.'.fields.name');
        $emailFieldPosition = 0;
        $nameFieldPosition = 1;

        $lines = [];
        $emails = [];
        $names = [];
        $errors = [];

        foreach ($rows as $index => $row) {
            $this->assertValideLine($row, $errors, $lines);

            if ($index==0) {
                $row = array_map('mb_strtolower', $row);
                $emailFieldPosition = collect($row)->search($emailFieldLabel);
                $nameFieldPosition = collect($row)->search($nameFieldLabel);
                continue;
            }

            $email = $row[$emailFieldPosition] ?? '';
            $name = $row[$nameFieldPosition] ?? '';

            $this->assertValidEmail($email, $errors, $emails);
            $this->assertValidName($name, $errors, $names);
        }
        $progressTrackerCallback($errors);
    }

    private function assertValideLine(array $row, array &$errors, array &$lines): void
    {
        $lineString = implode(',', $row);

        if (isset($lines[$lineString])) {
            $errors[] = Lang::get('validation.duplicated_line_detected', ['line' => $lineString]);
        }
        $lines[$lineString] = true;
    }

    private function assertValidEmail(string $email, array &$errors, array &$emails): void
    {
        if (empty(trim($email))) {
            $errors[] = Lang::get('validation.empty_email_detected');
        } elseif (isset($emails[$email])) {
            $errors[] = Lang::get('validation.duplicated_email_detected', ['email' => $email]);
        }
        $emails[$email] = true;

        if (config('app.secured_email_validation', true)) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $errors[] = Lang::get('validation.invalide_email_detected', [
                    'email' => $email
                ]);
            } else {
                try {
                    $this->validateData([
                        'email' => $email
                    ], [
                        'email' => ['required', 'email:rfc,dns'],
                    ]);
                } catch (\Exception $exception) {
                    $errors[] = Lang::get('validation.invalide_email_detected', [
                        'email' => $email
                    ]);
                }
            }
        }
    }

    private function assertValidName(string $name, array &$errors, array &$names): void
    {
        if (empty(trim($name))) {
            $errors[] = Lang::get('validation.empty_name_detected');
        } elseif (isset($names[$name])) {
            $errors[] = Lang::get('validation.duplicated_name_detected', [
                'name' => $name
            ]);
        }
        $names[$name] = true;
    }
}
