<?php

namespace Inside\CSV\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\CSV\Actions\FeedImport;
use Inside\CSV\Exceptions\FileNotFoundException;
use Inside\CSV\Traits\FeedConfig;

class UserImportService
{
    use FeedConfig;

    private const STEP_WEIGHT = 20;

    private int $progresBar = 0;
    private string $processTracker;

    public function __construct()
    {
    }

    public function import(string $file, bool $disableUsers, string $processTracker): void
    {
        try {
            $extension = $this->getExtension($file);
            $this->prepareImportFile($file, $extension);
            $this->processTracker = $processTracker;

            /**
             * @var FeedImport $importer
             */
            $importer = app(FeedImport::class);
            $this->updateProgress();

            $importer->import(
                fn ($statistics) => $this->handleImportationProcess($statistics),
                $extension,
                $disableUsers,
                $this->processTracker
            );
        } catch (\Exception | \Throwable $exception) {
            $this->addErrors([$exception->getMessage()]);
        } finally {
            Cache::tags([$this->processTracker])->put('state', 100, now()->addHours(2));
        }
    }


    private function handleImportationProcess(array $statistics): void
    {
        $this->updateProgress();
        if (!empty($statistics)) {
            $this->addImportStatistics($statistics);
        }
    }

    private function updateProgress(): void
    {
        $this->progresBar += self::STEP_WEIGHT;
        Cache::tags([$this->processTracker])->put('state', min(100, $this->progresBar), now()->addHours(2));
    }

    private function addErrors(array $errors): void
    {
        Cache::tags([$this->processTracker])->put('errors', $errors, now()->addHours(2));
    }

    private function addImportStatistics(array $statistics): void
    {
        Cache::tags([$this->processTracker])->put('statistics', $statistics, now()->addHours(2));
    }

    private function getExtension(string $file): string
    {
        if (!Storage::disk('local')->exists($file)) {
            throw new FileNotFoundException(
                Lang::get('validation.fail_open_file', [
                    'file' => basename($file)
                ]),
            );
        }

        $path = Storage::disk('local')->path($file);
        return str(pathinfo($path, PATHINFO_EXTENSION))->lower();
    }

    private function prepareImportFile(string $file, string $format): void
    {
        if (!Storage::disk('local')->exists($file)) {
            throw new FileNotFoundException(
                Lang::get('validation.fail_open_file', [
                    'file' => basename($file)
                ]),
            );
        }
        $filePath = Storage::disk('local')->path($file);
        $this->setFilePath($format, $filePath);
    }
}
