<?php

namespace Inside\CSV\Services;

use Faker\Factory;
use Faker\Generator;
use Illuminate\Support\Facades\Lang;
use Inside\CSV\Actions\TemplateExport\CsvExportStrategy;
use Inside\CSV\Actions\TemplateExport\ExcelExportStrategy;
use Inside\CSV\Actions\TemplateExport\ExportStrategy;
use Inside\CSV\Exceptions\HeadersMissingException;
use Inside\CSV\Exceptions\InvalidFormatException;
use Inside\CSV\Traits\FeedConfig;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class TemplateDownloadService
{
    use FeedConfig;

    public function __construct()
    {
    }

    public function downloadTemplate(string $format): BinaryFileResponse
    {
        $headers = $this->originalHeaders($format);

        if (empty($headers)) {
            throw new HeadersMissingException(
                Lang::get('validation.headers_empty_config')
            );
        }

        $faker = Factory::create();
        $exampleRow = array_map(
            fn ($fieldType) => $this->generateExampleValue($fieldType, $faker),
            $headers
        );

        $exporter = $this->getExporter($format);

        return $exporter->export(array_keys($headers), $exampleRow);
    }

    private function generateExampleValue(string $fieldType, Generator $faker): string
    {
        return match ($fieldType) {
            'mail' => $faker->safeEmail,
            'name' => $faker->name,
            'firstname' => $faker->firstName,
            'lastname' => $faker->lastName,
            'phone'  => $faker->phoneNumber,
            'function' => $faker->jobTitle,
            'adress' => $faker->address,
            'city' => $faker->city,
            'date' => $faker->date,
            'company' => $faker->company,
            'department' => $faker->domainWord,
            default => $faker->word,
        };
    }

    /**
     * @param string $format
     * @return ExportStrategy
     * @throws InvalidFormatException
     */
    private function getExporter(string $format): ExportStrategy
    {
        return match ($format) {
            'csv' => app(CsvExportStrategy::class),
            'xlsx' => app(ExcelExportStrategy::class),
            default => throw new InvalidFormatException(sprintf('Invalid format requested: %s', $format)),
        };
    }
}
