<?php

namespace Inside\CSV\Actions\FeedValidation;

use Illuminate\Support\Facades\Lang;
use Inside\Validation\ValidateRequests;

class CsvFeedValidationStrategy implements FeedValidationStrategy
{
    use ValidateRequests;

    private const VALID_EXTENSION = 'csv';

    public function validate(string $file, array $expectedColumns, callable $progressTrackerCallback): void
    {
        $this->assertValidFileType($progressTrackerCallback, $file);
        $isParsableFile = $this->assertUtf8Encoding($progressTrackerCallback, $file);
        if ($isParsableFile) {
            $this->assertValidSeparatorAndColumnCount($progressTrackerCallback, $file, $expectedColumns);
            $this->assertColumnNamesMatch($progressTrackerCallback, $file, $expectedColumns);
            $this->assertValidRows($progressTrackerCallback, $file);
        }
    }

    private function assertValidFileType(callable $progressTrackerCallback, string $file): void
    {
        $extension = pathinfo($file, PATHINFO_EXTENSION);
        if (self::VALID_EXTENSION !== $extension) {
            throw new \RuntimeException(
                Lang::get('validation.invalid_file_type', [
                    'type' => $extension,
                    'expected' => self::VALID_EXTENSION
                ])
            );
        }
        $progressTrackerCallback([]);
    }

    private function assertUtf8Encoding(callable $progressTrackerCallback, string $file): bool
    {
        $errors = [];
        /**
         * @var string $content
         */
        $content = file_get_contents($file);
        $encoding = mb_detect_encoding($content, 'UTF-8', true);

        if ($encoding == 'UTF-8') {
            $progressTrackerCallback($errors);
            return true;
        }

        if (!$encoding) {
            $errors[] = Lang::get('validation.invalid_encoding');
            $progressTrackerCallback($errors);
            return false;
        }

        $content = mb_convert_encoding($content, 'UTF-8', $encoding);
        if (mb_detect_encoding($content, 'UTF-8', true) !== 'UTF-8') {
            $errors[] = Lang::get('validation.invalid_encoding');
        } else {
            file_put_contents($file, $content);
        }

        $progressTrackerCallback($errors);
        return empty($errors);
    }

    private function assertValidSeparatorAndColumnCount(callable $progressTrackerCallback, string $file, array $expectedColumns): void
    {
        $code = config('app.code');
        $separator = config('csv.files.' . $code.'.separator');
        $handle = fopen($file, 'r');
        if (!$handle) {
            throw new \RuntimeException(
                Lang::get('validation.fail_open_file', [
                    'file' => basename($file)
                ]),
            );
        }

        while (($line = fgets($handle)) !== false) {
            $parsedLine = str_getcsv($line, $separator);
            if (count($parsedLine) < count($expectedColumns)) {
                fclose($handle);
                throw new \RuntimeException(
                    Lang::get('validation.invalid_separator')
                );
            }
        }

        fclose($handle);
        $progressTrackerCallback([]);
    }

    private function assertColumnNamesMatch(callable $progressTrackerCallback, string $file, array $expectedColumns): void
    {
        $errors = [];
        $code = config('app.code');
        $separator = config('csv.files.' . $code.'.separator');
        $lines = file($file);
        if (empty($lines)) {
            throw new \RuntimeException(Lang::get('validation.invalid_empty_file'));
        }

        $headers = array_map('trim', str_getcsv($lines[0], $separator));
        if (
            array_diff(
                array_map('mb_strtolower', $expectedColumns),
                array_map('mb_strtolower', $headers)
            )
        ) {
            $errors[] = Lang::get('validation.invalid_column_names');
            $errors[] = json_encode(array_diff(
                array_map('mb_strtolower', $expectedColumns),
                array_map('mb_strtolower', $headers)
            ));
        }

        if (count($headers) !== count(array_unique($headers))) {
            $errors[] = Lang::get('validation.duplicate_column_names');
        }

        $progressTrackerCallback($errors);
    }

    private function assertValidRows(callable $progressTrackerCallback, string $file): void
    {
        $handle = fopen($file, 'r');
        if (!$handle) {
            throw new \RuntimeException(
                Lang::get('validation.fail_open_file', [
                    'file' => basename($file)
                ]),
            );
        }

        $code = config('app.code');
        $emailFieldLabel = config('csv.files.' . $code.'.fields.mail') ?? config('csv.files.' . $code.'.fields.email');
        $nameFieldLabel = config('csv.files.' . $code.'.fields.name');
        $separator = config('csv.files.' . $code.'.separator');
        $emailFieldPosition = 0;
        $nameFieldPosition = 1;

        $lines = [];
        $emails = [];
        $names = [];
        $errors = [];
        $count = 0;

        do {
            $rows = fgetcsv($handle, null, $separator);
            if ($rows === false) {
                break;
            }

            $count++;

            $this->assertValideLine($rows, $errors, $lines);

            if ($count == 1) {
                $rows = array_map('mb_strtolower', $rows);
                $emailFieldPosition = collect($rows)->search($emailFieldLabel);
                $nameFieldPosition = collect($rows)->search($nameFieldLabel);
                continue;
            }

            $email = $rows[$emailFieldPosition] ?? '';
            $name = $rows[$nameFieldPosition] ?? '';

            $this->assertValidEmail($email, $errors, $emails);
            $this->assertValidName($name, $errors, $names);
        } while (true);

        fclose($handle);

        $errors = array_unique($errors);
        $progressTrackerCallback($errors);
    }

    private function assertValideLine(array $rows, array &$errors, array &$lines): void
    {
        $lineString = implode(',', $rows);

        if (isset($lines[$lineString])) {
            $errors[] = Lang::get('validation.duplicated_line_detected', ['line' => $lineString]);
        }
        $lines[$lineString] = true;
    }

    private function assertValidEmail(string $email, array &$errors, array &$emails): void
    {
        if (empty(trim($email))) {
            $errors[] = Lang::get('validation.empty_email_detected');
        } elseif (isset($emails[$email])) {
            $errors[] = Lang::get('validation.duplicated_email_detected', ['email' => $email]);
        }
        $emails[$email] = true;

        if (config('app.secured_email_validation', true)) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $errors[] = Lang::get('validation.invalide_email_detected', [
                    'email' => $email
                ]);
            } else {
                try {
                    $this->validateData([
                        'email' => $email
                    ], [
                        'email' => ['required', 'email:rfc,dns'],
                    ]);
                } catch (\Exception $exception) {
                    $errors[] = Lang::get('validation.invalide_email_detected', [
                        'email' => $email
                    ]);
                }
            }
        }
    }

    private function assertValidName(string $name, array &$errors, array &$names): void
    {
        if (empty(trim($name))) {
            $errors[] = Lang::get('validation.empty_name_detected');
        } elseif (isset($names[$name])) {
            $errors[] = Lang::get('validation.duplicated_name_detected', [
                'name' => $name
            ]);
        }
        $names[$name] = true;
    }
}
