<?php

namespace Inside\CSV\Actions;

use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\CSV\Events\ImportFinishedEvent;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

class FeedImport
{
    public function import(callable $progressTrackerCallback, string $format, bool $disableUsers, string $processTracker): void
    {
        $cmd = sprintf('php artisan inside:user:import %s --process-tracker=%s', $format, $processTracker);

        if ($disableUsers) {
            $cmd .= ' --disable-not-imported';
        }

        $process = Process::fromShellCommandline($cmd);
        $process->setTimeout(null);
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        $output = $process->getOutput();
        $statsFromOutput = $this->parsedOutput($output, $disableUsers);
        $progressTrackerCallback($statsFromOutput);

        event(new ImportFinishedEvent(
            $statsFromOutput['total'],
            $statsFromOutput['imported'],
            $statsFromOutput['disabled'],
            App::runningInConsole() ? User::where('email', config('app.technical_mail'))->first() : Auth::user()
        ));
    }

    private function parsedOutput(string $output, bool $disableUsers): array
    {
        $usersFound = (int) str()->between(str()->before($output, 'users found'), '[', ']');
        $importedUsers = (int) str()->between(str()->after($output, 'Total users imported:'), '[', ']');

        $disabledUsers =  !$disableUsers || !str()->contains($output, 'Disabling activated users')
            ? 0
            : (int) str()->between(str()->after($output, 'Total users disabled:'), '[', ']');

        $activeUsersNumber = type_to_class('users')::where('status', 1)->count();
        $ignored = $activeUsersNumber - $importedUsers;

        return [
            'total' => $usersFound,
            'imported' => $importedUsers,
            'disabled' => $disabledUsers,
            'ignored' => max(0, $ignored)
        ];
    }
}
