<?php

namespace Inside\AzureAD\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class AzureADService
{
    private function __construct(
        protected AzureADAuth $authentication,
        public bool $is_microsoft_api,
        public array $custom_attributes,
        public array $custom_security_attributes,
        public ?string $group_id
    ) {
    }

    public static function load(array $entry): self
    {
        return new self(
            AzureADAuth::load($entry),
            $entry['microsoft_api'],
            $entry['custom_attributes'],
            $entry['custom_security_attributes'],
            $entry['group_id'],
        );
    }

    /**
     * Setup filter or field selection
     *
     * @param string $filter
     * @param bool $microsoft
     * @param string|null $skiptoken
     * @param int $limit
     * @return string
     */
    protected function prepareFilter(string $filter, bool $microsoft, ?string $skiptoken, int $limit): string
    {
        $query = '';
        if ($filter) {
            $query = (($microsoft ? '?' : '&') . '$count=true&$filter=' . rawurlencode($filter));
            if (isset($skiptoken)) {
                $query .= '&$skiptoken=' . $skiptoken;
            }
        } elseif ($limit > 0) {
            $query = ($microsoft ? '?' : '&') . '$top=' . $limit;
        } elseif (isset($skiptoken)) {
            $query = ($microsoft ? '?' : '&') . '$skiptoken=' . $skiptoken;
        }

        // we expand some fields
        if ($expand = $this->authentication->expand) {
            $query .= empty($query) ? '?' : '';
            $query .= '&expand=' . $expand;
        }

        return $query;
    }


    /**
     *
     * return url depending on which api we use
     *
     * @param string $resource
     * @param string $filter
     * @param string|null $skiptoken
     * @param int $limit
     * @return string
     */
    protected function prepareUrl(string $resource, string $filter, ?string $skiptoken = null, int $limit = 0): string
    {
        // build the microsoft url
        if ($this->is_microsoft_api) {
            return sprintf(
                '%s/%s/%s/%s%s',
                $this->authentication->graph_url,
                $this->authentication->api_version,
                $this->authentication->app_tenant,
                $resource,
                $this->prepareFilter($filter, true, $skiptoken, $limit)
            );
        }

        return sprintf(
            '%s/%s/%s?api-version=%s%s',
            $this->authentication->graph_url,
            $this->authentication->app_tenant,
            $resource,
            $this->authentication->api_version,
            $this->prepareFilter($filter, false, $skiptoken, $limit)
        );
    }

    /**
     * Get data for a resource
     *
     * @link   https://docs.microsoft.com/fr-fr/azure/active-directory/develop/active-directory-graph-api-quickstart
     *         Documentation
     * @param String $resource
     * @param string $filter
     * @param null|string $skiptoken
     * @param int $limit
     * @todo   Catch errors
     * @return mixed|null
     */
    public function search(String $resource, string $filter = '', ?string $skiptoken = null, int $limit = 0)
    {
        try {
            $accessToken = $this->authentication->getAccessToken();
        } catch (\Exception $exception) {
            Log::error(
                '[AzureAD::search] failed {' . $exception->getMessage() . '}'
            );
            throw new \Exception($exception->getMessage());
        }
        $curl = curl_init();

        $header = [
            'Authorization: Bearer ' . $accessToken
        ];
        if ($filter) {
            $header[] = 'ConsistencyLevel: eventual';
        }
        curl_setopt_array(
            $curl,
            [
                CURLOPT_RETURNTRANSFER => 1,

                CURLOPT_URL            => $this->prepareUrl($resource, $filter, $skiptoken, $limit),
                CURLOPT_HTTPHEADER     => $header
            ]
        );

        $data     = curl_exec($curl);

        $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if (!is_string($data)) {
            Log::error('[AzureAD::search] curl call returned false.');
            return null;
        }

        if ($httpcode != 200) {
            $error = json_decode($data);
            Log::error(
                '[AzureAD::search] failed {' . $error->{'odata.error'}->message->value . '} (' . $httpcode . ')'
            );

            throw new \Exception('[AZUREAD::search][ERROR_CODE::'.$error->error->code.'] '.$error->error->message.'');
        }

        return json_decode($data);
    }

    /**
     * Get user info
     *
     * @param mixed $user an azure ad user object
     * @return mixed|null
     */
    public function user($user)
    {
        $accessToken = $this->authentication->getAccessToken();

        $curl = curl_init();

        curl_setopt_array(
            $curl,
            [
                CURLOPT_RETURNTRANSFER => 1,
                CURLOPT_URL => $this->prepareUrl('users/' . $user->userPrincipalName, ''),
                CURLOPT_HTTPHEADER     => ['Authorization: Bearer ' . $accessToken],
            ]
        );

        $data     = curl_exec($curl);

        $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if (!is_string($data)) {
            Log::error('[AzureAD::user] curl call returned false.');
            return null;
        }

        if ($httpcode != 200) {
            $error = json_decode($data);
            Log::error('[AzureAD::user] failed {' . $error->{'odata.error'}->message->value . '}');

            return null;
        }

        return json_decode($data);
    }

    /**
     * Get user photo
     *
     * @param mixed $user an azure ad user object
     * @return mixed|null
     */
    public function photo($user)
    {
        $accessToken = $this->authentication->getAccessToken();

        $options  = ['http' => [
            'header' => "Authorization: Bearer $accessToken"
        ]];
        $context  = stream_context_create($options);

        $apiUrl = sprintf(
            '%s/%s/users/%s/photo/$value',
            $this->authentication->graph_url,
            $this->authentication->api_version,
            $user->userPrincipalName
        );

        $image = file_get_contents($apiUrl, false, $context);
        if (!$image) {
            Log::error('[AzureAD::photo] error occurred while trying to get the profile image.');
            return null;
        }

        $fileName   = Str::uuid();
        $chunkId    = Str::random(32);
        $chunkPath  = "chunks/$chunkId/$fileName.jpeg";
        Storage::makeDirectory("chunks/$chunkId");
        Storage::disk('local')->put($chunkPath, $image);
        return $chunkPath;
    }

    /**
     * @param string $url
     * @return mixed|null
     */
    protected function get(string $url)
    {
        $accessToken = $this->authentication->getAccessToken();

        $curl = curl_init();

        curl_setopt_array(
            $curl,
            [
                CURLOPT_RETURNTRANSFER => 1,
                CURLOPT_URL            => $url,
                CURLOPT_HTTPHEADER     => ['Authorization: Bearer ' . $accessToken],
            ]
        );

        $data     = curl_exec($curl);
        $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if (!is_string($data)) {
            Log::error('[AzureAD::get] curl call returned false.');
            return null;
        }

        if ($httpcode != 200) {
            $error = json_decode($data);
            Log::error('[AzureAD::get] failed {' . $error->{'odata.error'}->message->value . '} [' . $url . ']');

            return null;
        }
        return json_decode($data);
    }

    /**
     * @param string $accessToken
     * @return mixed|null
     */
    public function me(string $accessToken)
    {
        $curl = curl_init();

        $azure = $this->authentication->graph_url . '/me?api-version=' . $this->authentication->api_version;
        $microsoft = $this->authentication->graph_url . '/' . $this->authentication->api_version . '/me';

        $url = $this->is_microsoft_api ? $microsoft : $azure;

        curl_setopt_array(
            $curl,
            [
                CURLOPT_RETURNTRANSFER => 1,
                CURLOPT_URL            => $url,
                CURLOPT_HTTPHEADER     => ['Authorization: Bearer ' . $accessToken],
            ]
        );

        $data     = curl_exec($curl);
        $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if (!is_string($data)) {
            Log::error('[AzureAD::me] curl call returned false.');
            return null;
        }

        if ($httpcode != 200) {
            $error = json_decode($data);
            Log::error('[AzureAD::me] failed {' . $error->{'odata.error'}->message->value . '}');

            return null;
        }

        return json_decode($data);
    }
}
