<?php

namespace Inside\AzureAD\Services;

use Illuminate\Support\Facades\Log;

/**
 * Class Authenticator
 *
 * @package Inside\AzureAD\Services
 */
class AzureADAuth
{
    // TODO: PHP8.1 -> READONLY
    private function __construct(
        public string $app_id,
        public string $app_tenant,
        public string $secret_key,
        public string $graph_url,
        public string $oauth2_url,
        public string $api_version,
        public ?string $expand,
        public ?string $cert_private_key_base64,
        public ?string $cert_thumbprint,
    ) {
    }

    public static function load(array $entry): self
    {
        return new self(
            $entry['app_id'],
            $entry['app_tenant'],
            $entry['secret_key'],
            $entry['graph_url'],
            $entry['oauth2']['url'],
            $entry['api_version'],
            $entry['expand'],
            $entry['cert_private_key_base64'],
            $entry['cert_thumbprint'],
        );
    }

    /**
     * Get new access token (expires after 1H)
     * @link https://docs.microsoft.com/fr-fr/azure/active-directory/develop/v1-oauth2-client-creds-grant-flow#request-an-access-token Documentation
     * @todo Catch errors
     * @return mixed
     */
    public function getAccessToken()
    {
        $tokenUrl = $this->oauth2_url . '/' . $this->app_tenant . '/oauth2/token';

        if ($this->cert_private_key_base64) {
            try {
                return $this->getAccessTokenWithCertificate($tokenUrl);
            } catch (\Throwable $e) {
                Log::error('[AzureADAuth::getAccessToken] Cert auth failed, falling back to client_secret: ' . $e->getMessage());
                throw new \Exception('[AzureADAuth::getAccessToken] Cert auth failed, falling back to client_secret: ' . $e->getMessage());
            }
        }
        return $this->getAccessTokenWithClientSecret($tokenUrl);
    }

    private function getAccessTokenWithClientSecret(string $tokenUrl): string
    {
        $postFields = [
            'grant_type'    => 'client_credentials',
            'client_id'     => $this->app_id,
            'client_secret' => $this->secret_key,
            'resource'      => $this->graph_url,
        ];

        $response = $this->doRequest($tokenUrl, $postFields);

        if (!is_string($response['data'])) {
            Log::error('[AzureADAuth::getAccessToken] curl call returned false.');
            throw new \Exception('[AzureADAuth::getAccessToken] curl call returned false.');
        }

        $data = json_decode($response['data']);

        if ($response['httpcode'] != 200) {
            Log::error('[AzureAD::getAccessToken] ' . $data->error_description);
            throw new \Exception('[AzureAD::getAccessToken]' . $data->error_description . '');
        }

        return $data->access_token;
    }

    private function getAccessTokenWithCertificate(string $tokenUrl): string
    {
        /** @phpstan-ignore-next-line */
        $privateKeyPem = base64_decode($this->cert_private_key_base64, true);
        if ($privateKeyPem === false) {
            throw new \RuntimeException('Invalid Base64 private key.');
        }

        $privateKey = openssl_pkey_get_private($privateKeyPem, '');
        if ($privateKey === false) {
            throw new \RuntimeException('Cannot load certificate private key (bad password or invalid key).');
        }

        $clientAssertion = $this->buildClientAssertion($tokenUrl, $privateKey);

        $postFields = [
            'grant_type'            => 'client_credentials',
            'client_id'             => $this->app_id,
            'resource'              => $this->graph_url,
            'client_assertion_type' => 'urn:ietf:params:oauth:client-assertion-type:jwt-bearer',
            'client_assertion'      => $clientAssertion,
        ];

        $response = $this->doRequest($tokenUrl, $postFields);

        if (!is_string($response['data'])) {
            Log::error('[AzureADAuth::getAccessTokenWithCertificate] curl call returned false.');
            throw new \RuntimeException('AzureAD certificate auth failed: curl error.');
        }

        $data = json_decode($response['data']);

        if ($response['httpcode'] !== 200) {
            $msg = $data->error_description ?? 'Unknown error';
            Log::error('[AzureADAuth::getAccessTokenWithCertificate] ' . $msg);
            throw new \RuntimeException('[AzureADAuth::getAccessTokenWithCertificate] ' . $msg);
        }

        return $data->access_token;
    }
    private function doRequest(string $url, array $postFields): array
    {
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_URL            => $url,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $postFields,
        ]);

        $data = curl_exec($curl);
        $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        return [
            'data' => $data,
            'httpcode' => $httpcode,
        ];
    }

    private function buildClientAssertion(string $tokenUrl, \OpenSSLAsymmetricKey $privateKey): string
    {
        $now = time();

        $header = [
            'alg' => 'RS256',
            'typ' => 'JWT',
        ];

        if ($this->cert_thumbprint) {
            $hex = str_replace([' ', ':'], '', $this->cert_thumbprint);
            $bin = hex2bin($hex);
            if ($bin !== false) {
                $header['x5t'] = $this->base64UrlEncode($bin);
            }
        }

        $payload = [
            'aud' => $tokenUrl,
            'iss' => $this->app_id,
            'sub' => $this->app_id,
            'jti' => bin2hex(random_bytes(16)),
            'nbf' => $now,
            'exp' => $now + 3600,
        ];

        $jsonHeader = json_encode($header);
        if ($jsonHeader === false) {
            throw new \RuntimeException('Failed to encode JWT header into JSON.');
        }

        $jsonPayload = json_encode($payload);
        if ($jsonPayload === false) {
            throw new \RuntimeException('Failed to encode JWT payload into JSON.');
        }

        $jwtHeader  = $this->base64UrlEncode($jsonHeader);
        $jwtPayload = $this->base64UrlEncode($jsonPayload);
        $signingInput = $jwtHeader . '.' . $jwtPayload;

        $signature = '';
        if (!openssl_sign($signingInput, $signature, $privateKey, OPENSSL_ALGO_SHA256)) {
            throw new \RuntimeException('Failed to sign client assertion with private key.');
        }

        $jwtSignature = $this->base64UrlEncode($signature);

        return $signingInput . '.' . $jwtSignature;
    }

    private function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
}
