<?php

namespace Inside\AzureAD\AzureAD;

use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Inside\AzureAD\Services\AzureADService;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;

class AzureADImporter implements ImporterInterface
{
    /**
     * Function supposed to initialize everything for the import
     *
     * @return bool
     */
    public function prepareImport(): bool
    {
        // no global operation to do.
        chdir(dirname(base_path(), 3));

        return true;
    }

    /**
     * Return user data after synchronized attribute with the type of import (syn_attributes in the config file)
     *
     * @param array $user
     * @param array $entry
     * @param BridgeContent $bridge
     * @return array
     */
    public function getSyncAttributes($user, $entry, $bridge): array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        $syncAttributes = $entry['sync_attributes'] ?? [
                'email' => 'mail',
                'name'  => 'name',
            ];
        // Note: charles indique d'utiliser l'upn et non l'id
        $discoveringAttribute = $entry['discovering_attribute'] ?? 'userPrincipalName';
        if (!isset($syncAttributes[$discoveringAttribute])) {
            $syncAttributes[$discoveringAttribute] = $discoveringAttribute;
        }
        Log::channel('import')->info('[AzureAD] got user {' . json_encode($user) . '}');
        $data = [];

        $azure = AzureADService::load($entry);

        if (!empty($azure->custom_attributes)) {
            $user = $this->handleCustomAttributes($azure, $user);
        }

        if (!empty($azure->custom_security_attributes)) {
            $user = $this->handleCustomSecurityAttributes($azure, $user);
        }

        foreach ($syncAttributes as $modelField => $azureadField) {
            if ($modelField === 'image') {
                $data[$modelField] = $azure->photo($user);
            } elseif (is_object($azureadField) && ($azureadField instanceof \Closure)) {
                $data[$modelField] = $azureadField($user, $bridge);
                if ($data[$modelField] === null) {
                    unset($data[$modelField]);
                }
            } elseif (isset($user->{$azureadField})) {
                $data[$modelField] = $user->{$azureadField};
            } else {
                $data[$modelField] = null;
            }
        }

        $postSync = config('azuread.post_sync', false);

        if (is_object($postSync) && $postSync instanceof \Closure) {
            $postSync($data);
        }

        return $data;
    }

    /**
     * handle custom attributes
     * Exemple :
     *  'custom_attributes' => [
            'userType' => 'value',
            'manager' => 'email'
        ],
     * will add an userType and a manager key and value by retrieving them directly from the users
     *
     * @param AzureADService $azure
     * @param mixed $user
     * @return mixed
     */
    public function handleCustomAttributes(AzureADService $azure, $user)
    {
        foreach ($azure->custom_attributes as $customKey => $customValue) {
            if (empty($customKey)) {
                continue;
            }
            $upn = $user->userPrincipalName;
            Log::channel('import')->info("Processing custom attribute $customKey for user $upn");
            try {
                $result = $azure->search('users/' . $upn . '/' . $customKey, '', null);
            } catch (\Exception $e) {
                Log::channel('import')->info("No $customKey found for $upn");
                continue;
            }
            if ($result) {
                $value = $result->{$customValue};
                $user->{$customKey} = $value ?? null;
            }
        }
        return $user;
    }

    /**
     * handle custom security attributes
     * Exemple :
     *  'custom_security_attributes' => [
        'inside' => 'birthday',
        'attribute_set' => 'attribute_name'
     ],
     *
     * @param AzureADService $azure
     * @param mixed $user
     * @return mixed
     */
    public function handleCustomSecurityAttributes(AzureADService $azure, $user)
    {
        $upn = $user->userPrincipalName;

        try {
            Log::channel('import')->info("Processing custom security attribute for user $upn");
            $result = $azure->search('users/' . $upn . '?$select=customSecurityAttributes', '', null);
            $result = $result->customSecurityAttributes;
        } catch (\Exception $e) {
            Log::channel('import')->info("No custom security attribute found for $upn");
            return $user;
        }

        foreach ($azure->custom_security_attributes as $attributeSet => $attributeName) {
            $value = $result?->{$attributeSet}?->{$attributeName};

            $user->{$attributeName} = $value;
        }

        return $user;
    }

    /**
     * get the users from azure
     *
     * @param AzureADService $azure
     * @param string $resource
     * @param string|null $filter
     * @param int $limit
     * @return Collection
     */
    public function getUsersFromAzure(AzureADService $azure, string $resource, ?string $filter, int $limit): Collection
    {
        $users = collect();

        $next = null;
        do {
            $nextLink = $azure->is_microsoft_api ? '@odata.nextLink' : 'odata.nextLink';
            // get only users from a group or get all users
            try {
                $result = $azure->search(resource: $resource, filter: $filter ?? '', skiptoken: $next, limit: $limit);
            } catch (\Exception $exception) {
                Log::channel('import')->info($exception->getMessage());
                throw new \Exception($exception->getMessage());
            }


            if ($result) {
                $users = $users->concat($result->value);
                if (isset($result->{$nextLink})) {
                    $pagination = parse_url($result->{$nextLink}, PHP_URL_QUERY);
                } else {
                    $pagination = null;
                }

                $queries = [];

                if (is_string($pagination)) {
                    parse_str($pagination, $queries);
                }

                $next = $queries['$skiptoken'] && empty($queries['$top']) ? $queries['$skiptoken'] : null;
            } else {
                $next = null;
            }
        } while ($next !== null);

        $postFilter = config('azuread.postfilter');
        if ($postFilter && is_callable($postFilter)) {
            $users = $postFilter($users);
        }

        return $users;
    }

    /**
     * Get members from all groups
     *
     * @param AzureADService $azure
     * @param array $ids
     * @param string|null $filter
     * @param int $limit
     * @return Collection
     */
    public function getGroupMembers(AzureADService $azure, array $ids, ?string $filter, int $limit): Collection
    {
        $users = collect();
        foreach ($ids as $id) {
            $users = $users->concat($this->getUsersFromAzure($azure, 'groups/' . trim($id) . '/members', $filter, $limit));
        }
        return $users;
    }

    /**
     * Get all users from the base provider (google, csv file, etc.)
     *
     * @param array $entry
     * @param string|null $filter
     * @param array|null $test
     * @param int $limit
     * @return array
     */
    public function getUsers($entry, $filter, $test = null, int $limit = 0): array
    {
        if ($test) {
            return [(object)$test];
        }

        $users = collect();

        $azure = AzureADService::load($entry);
        try {
            if ($groupIds = $azure->group_id) {
                $ids = explode(',', $groupIds);
                $users = $this->getGroupMembers($azure, $ids, $filter, $limit);
            } else {
                $users = $this->getUsersFromAzure($azure, 'users', $filter, $limit);
            }
        } catch (\Exception $exception) {
            throw new \Exception($exception->getMessage());
        }
        return $users->toArray();
    }

    /**
     * Get all entries from the base provider
     * (example: for google, one entry is a domain)
     * (example: for csv, one entry is one file)
     *
     * if there is no several entry, must return ['default' => entry] (something like that)
     *
     * @return array
     */
    public function getEntries(): array
    {
        $defaultConfig = config('azuread');
        $domainsConfigs = $defaultConfig['azuread_domains'] ?? [];
        unset($defaultConfig['azuread_domains']);

        if (empty($domainsConfigs)) {
            return ['default' => config('azuread')];
        };

        $entries = [];
        foreach ($domainsConfigs as $domain => $configs) {
            $entries[$domain] = array_merge($defaultConfig, $configs);
        }

        return $entries;
    }

    /**
     * Get the function cleanData (launched after retrieving data for a given user) from config file
     * Signature must be : function(&$data)
     *
     * @return mixed
     */
    public function getCleanData()
    {
        return config('azuread.cleandata');
    }

    /**
     * Get the function preFlight (launched before the import) from the config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPreFlight()
    {
        return config('azuread.preflight', null);
    }

    /**
     * Get the function postFlight (launched after the import) from config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPostFlight()
    {
        return config('azuread.postflight', null);
    }

    /**
     * Get filters (from .env or config files, depends of import type)
     *
     * @return null|string
     */
    public function getFilter(): ?string
    {
        return config('azuread.filter', null);
    }

    /**
     * @param string|null $email
     * @param array $entry
     * @return array|null
     */
    public function getUserInformation($email, $entry): ?array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        if ($email) {
            $result = AzureADService::load($entry)->search('users', 'mail eq \'' . $email . '\'');

            return ($result->value && count($result->value) > 0) ? (array)$result->value[0] : null;
        }

        $users = $this->getUsers($entry, null);

        return (array)Arr::first($users);
    }
}
