<?php

namespace Inside\Twitter\Helpers;

class TwitterFeeder
{
    public const URL_API = 'https://api.twitter.com/oauth2/token';

    /**
     * @var string
     */
    private $key;

    /**
     * @var string
     */
    private $secret;

    /**
     * @var array|null
     */
    private $token;

    public function __construct()
    {
        $this->key = config('twitter.consumer_key');
        $this->secret = config('twitter.consumer_secret');
        $this->token = config('twitter.access_token');

        if ($this->token) {
            $this->token = json_decode($this->token, true);

            return;
        }

        if (empty($this->token)) {
            $this->authenticate();
        }
    }

    /**
     * @return void
     */
    private function authenticate(): void
    {
        $credentials = base64_encode($this->key . ':' . $this->secret);
        $headers = [
            'Authorization: Basic ' . $credentials,
            'Content-type: application/x-www-form-urlencoded;charset=UTF-8',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, self::URL_API);
        curl_setopt($ch, CURLOPT_PROXY, config('twitter.proxy', false));
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=client_credentials');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, config('twitter.ssl_verifypeer', true));
        $pre_token = curl_exec($ch);
        $this->token = json_decode((string)$pre_token, true);
    }

    /**
     * @param array $array
     * @param array $path
     * @return array
     */
    private function getValueKeyPath(array $array, array $path = []): array
    {
        $result = [];
        foreach ($array as $key => $val) {
            $currentPath = array_merge($path, [$key]);

            if (is_array($val)) {
                $result = array_merge($result, $this->getValueKeyPath($val, $currentPath));
            } else {
                $result[] = join('->', $currentPath);
            }
        }

        return $result;
    }

    /**
     * @param array $array
     * @param array $keys
     * @param mixed $value
     * @return void
     */
    private function assign(array &$array, array $keys, $value): void
    {
        $last_key = array_pop($keys);
        foreach ($keys as $key) {
            if (!isset($array[$key]) || !is_array($array[$key])) {
                $array[$key] = [];
            }
            $array = &$array[$key];
        }
        $array[$last_key] = $value;
        unset($array);
    }

    /**
     * @param string $url
     * @return string
     */
    private function getDatasFromUrl(string $url): string
    {
        $datas = '';
        if (isset($this->token['token_type']) && $this->token['token_type'] == 'bearer') {
            $headers = [
                'Authorization: Bearer ' . $this->token['access_token'],
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_PROXY, config('twitter.proxy', false));
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, config('twitter.ssl_verifypeer', true));

            $datas = curl_exec($ch);
        }

        return (string) $datas;
    }

    /**
     * @param array $feed
     * @return array $out
     */
    private function fromTextToUrl(array $feed): array
    {
        foreach ($feed as &$tweet) {
            if (isset($tweet['full_text'])) {
                $tweet['text'] = $tweet['full_text'];
                unset($tweet['full_text']);
            }

            if (isset($tweet['entities'])) {
                if (isset($tweet['entities']['urls'])) {
                    $urls = $tweet['entities']['urls'];
                    foreach ($urls as $url) {
                        if (isset($url['expended_url'])) {
                            $tweet['text'] = str_replace(
                                $url['url'],
                                '<a href="' . $url['expanded_url'] . '" target="_blank">' . $url['url'] . '</a>',
                                $tweet['text']
                            );
                        }
                    }
                }

                if (isset($tweet['entities']['hashtags'])) {
                    $hashtags = $tweet['entities']['hashtags'];
                    foreach ($hashtags as $hashtag) {
                        if (isset($hashtag['text'])) {
                            $tweet['text'] = str_replace(
                                '#' . $hashtag['text'],
                                '<a href="https://twitter.com/hashtag/' . $hashtag['text']
                                . '?src=hash" target="_blank">#' . $hashtag['text'] . '</a>',
                                $tweet['text']
                            );
                        }
                    }
                }

                if (isset($tweet['entities']['user_mentions'])) {
                    $users = $tweet['entities']['user_mentions'];
                    foreach ($users as $user) {
                        if (isset($user['screen_name'])) {
                            $tweet['text'] = str_replace(
                                '@' . $user['screen_name'],
                                '<a href="https://twitter.com/' . $user['screen_name'] . '?src=hash" target="_blank">@'
                                . $user['screen_name'] . '</a>',
                                $tweet['text']
                            );
                        }
                    }
                }

                $tweet['text'] = preg_replace(
                    '/(https:\/\/t.co\/[a-zA-Z0-9\-\.]+)/',
                    '<a href="$0" target="_blank">$0</a>',
                    $tweet['text']
                );
            }
        }

        return $feed;
    }

    /**
     * @param string $url
     * @param array $fields
     * @return array
     */
    public function feed(string $url, array $fields = []): array
    {
        // TODO : function need refacto
        $out = [];
        $whichEndpoint = array_values(array_slice(explode('/', $url), -1))[0];
        $whichEndpoint = substr($whichEndpoint, 0, strlen('tweets.json'));

        mb_internal_encoding('UTF-8');
        if (empty($fields)) {
            $datas = json_decode($this->getDatasFromUrl($url), true, JSON_UNESCAPED_UNICODE);

            if (empty($datas)) {
                return [];
            }

            if ($whichEndpoint === 'tweets.json') {
                $datas = $datas['statuses'];
            }

            return $this->fromTextToUrl($datas);
        }

        $datas = json_decode($this->getDatasFromUrl($url), true);
        if (empty($datas)) {
            return [];
        }

        if ($whichEndpoint === 'tweets.json') {
            $datas = $datas->statuses;
        }

        $paths = $this->getValueKeyPath($fields);
        foreach ($datas as $key => $value) {
            foreach ($paths as $path) {
                $copy = $value;
                $arr  = explode('->', $path);
                foreach ($arr as $item) {
                    if (is_array($copy)) {
                        foreach ($copy as &$val) {
                            if (isset($val->{$item})) {
                                $val = $val->{$item};
                            }
                        }
                    } else {
                        if (!isset($copy->{$item})) {
                            break;
                        } else {
                            $copy = $copy->{$item};
                        }
                    }
                    if ($item == end($arr)) {
                        $this->assign($out[$key], $arr, $copy);
                    }
                }
            }
        }

        return $this->fromTextToUrl($out);
    }
}
