<?php

namespace Inside\Twitter\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Inside\Twitter\Helpers\TwitterFeeder;

class TwitterController
{
    /**
     * @var string $url
     */
    private $url = 'https://api.twitter.com/1.1/statuses/user_timeline.json';

    /**
     * @param Request $request
     * @return void
     */
    private function applyParameters(Request $request): void
    {
        if ($request->has('search') && $request->input('search')) {
            $param = '?screen_name=';

            if ((array_values(array_slice(explode('/', $this->url), -1))[0]) === 'tweets.json') {
                $param = '?q=';
            }
            $this->url .= $param . urlencode($request->all()['search']);
        }

        if ($request->has('count') && $request->input('count')) {
            if (substr($this->url, -1) != '?') {
                $this->url .= '&';
            }
            $this->url .= 'count=' . urlencode($request->all()['count']);
        }

        if (config('twitter.exclude_replies')) {
            if (substr($this->url, -1) != '?') {
                $this->url .= '&';
            }
            $this->url .= '&exclude_replies=1';
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function view(Request $request): JsonResponse
    {
        $fields = [];
        if ($request->has('fields') && $request->input('fields')) {
            $fields = json_decode($request->all()['fields'], true);
        }

        if ($request->has('endpoint') && $request->input('endpoint')) {
            $this->url = is_string($request->input('endpoint')) ? $request->input('endpoint') : '';
        }
        $this->applyParameters($request);
        if (config('twitter.cache.enabled', true) === false) {
            $feeder = new TwitterFeeder();

            return response()->json($feeder->feed($this->url, $fields));
        }

        $key = md5(config('app.code') . '-twitter-' . $this->url);
        if (!Cache::store(config('twitter.cache.store'))->has($key)) {
            try {
                $feeder = new TwitterFeeder();
                Cache::store(config('twitter.cache.store'))->put(
                    $key,
                    $feeder->feed($this->url, $fields),
                    config('twitter.cache.lifetime')
                );
            } catch (\Exception $e) {
                Log::error('[Inside twitter] failed to load twits [' . $e->getMessage() . ']');
            }
        }

        return response()->json(Cache::store(config('twitter.cache.store'))->get($key));
    }
}
