<?php

namespace Inside\TV\Services;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Cache;
use Inside\Content\Models\Contents\TvWeathers;
use Inside\Facades\Http;

class OpenWeatherMapService
{
    private string $city;
    private string $country;
    private const OPEN_WEATHER_API_ENDPOINT = 'http://api.openweathermap.org/data/2.5/weather?units=metric&q=%s&appid=%s';

    public function __construct(private string $apiKey)
    {
    }

    public function setCity(string $city): self
    {
        $this->city = $city;
        return $this;
    }

    public function setCountry(string $country): self
    {
        $this->country = $country;
        return $this;
    }

    public function getWeather(): JsonResponse
    {
        if (empty($this->city)) {
            return response()->json([], 204);
        }

        $query = $this->buildQuery();

        $weatherData = $this->fetchWeatherData($query);

        return response()->json($weatherData);
    }

    protected function buildQuery(): string
    {
        $query = urlencode($this->city);

        if (!empty($this->country)) {
            $query .= ',' . urlencode($this->country);
        }

        return $query;
    }

    protected function fetchWeatherData(string $query): array
    {
        $cachedData = Cache::tags(['weather'])->get($query);

        if ($cachedData !== null) {
            return $cachedData;
        }

        try {
            $response = Http::get(
                sprintf(self::OPEN_WEATHER_API_ENDPOINT, $query, $this->apiKey)
            );
            $weatherData = $response->json();
            Cache::tags(['weather'])->put($query, $weatherData, now()->addMinutes(60));
        } catch (\Exception $e) {
            $weatherData = [];
        }

        return $weatherData;
    }
}
