<?php

namespace Inside\TV\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Authentication\Events\AuthenticationLoginEvent;
use Inside\Authentication\Models\User;
use Inside\Authentication\Services\AuthenticationService;
use Inside\Permission\Facades\Role;
use Inside\TV\Services\TvIpService;
use Inside\TV\Services\TvTokenService;
use Inside\TV\Services\TvUserService;
use Laravel\Lumen\Routing\Controller;

class TvController extends Controller
{
    public function __construct(
        private TvTokenService $tvTokenService,
        private TvIpService $tvIpService,
        private TvUserService $tvUserService,
        private AuthenticationService $authenticationService
    ) {
    }

    public function loginAsTv(Request $request): JsonResponse
    {
        if (! $this->authorized($request)) {
            return response()->json(['error' => 'Not authorized.'], 403);
        }

        /**
         * @var User $userTv
         */
        $userTv = $this->getTvUser();
        $token = $this->authenticationService->logAs($userTv, 'inside-tv', true, true);
        $request->headers->set('api_token', $token->plainTextToken);
        $this->authenticationService->setUserToMagicCookie($request, $userTv, $token->plainTextToken, true);
        $this->authenticationService->updateMagieCookieLifetime($request);

        $user = $token->accessToken->user;
        $transformed = $user->toArray();
        $transformed['api_token'] = $token->plainTextToken;
        $transformed['permissions'] = [
            'roles' => Role::listUserRoles($user),
            'add' => [],
            'custom' => [],
            'backoffice_entries' => [],
        ];
        AuthenticationLoginEvent::dispatch($user, $token->accessToken);

        return response()->json(['data' => $transformed]);
    }

    private function getTvUser(): ?User
    {
        return $this->tvUserService->getUser();
    }

    private function authorized(Request $request): bool
    {
        $token = $request->input('token', '');
        $ip = str($request->ip());
        $userTv = $this->getTvUser();
        $isAllowed = $this->tvTokenService->exists($token);
        $isAllowed &= $this->tvIpService->isWhitelisted($ip);
        $isAllowed &= $userTv !== null && $userTv->hasAllRoles(config('tv.role_tv'));

        return (bool) $isAllowed;
    }
}
