<?php

namespace Inside\TV\Services;

use Illuminate\Support\Facades\Storage;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContent;

final class DuplicateScreenTvService
{
    public function duplicate(string $sourceUuid, string $uuid): void
    {
        $modules = collect(config('tv.modules'))->mapWithKeys(
            fn ($module) => [
                $module => str($module)->camel()
                    ->ucfirst()
                    ->prepend('reverse')
                    ->toString()
            ]
        );

        $withParams = $modules->values()->toArray();

        /**
         * @var Content $sourceScreenTv
         */
        $sourceScreenTv = type_to_class('tv_screens')::with($withParams)
            ->where('uuid', $sourceUuid)
            ->first();

        $modules->each(
            fn ($relation, $contentType) => $this->checkForDulicableContentsForContentType(
                $uuid,
                $relation,
                $contentType,
                $sourceScreenTv
            )
        );
    }

    private function checkForDulicableContentsForContentType(
        string $tvScrennUuid,
        string $relation,
        string $contentType,
        Content $sourceScreenTv
    ): void {
        $relatedContentTypes = collect(Schema::getRelations($contentType))
            ->diff(['authors','tvScreens'])->values()->toArray();

        $sourceScreenTv->{$relation}->where('status', 1)->each(
            fn ($content) => $this->duplicateContent($content, $relatedContentTypes, $tvScrennUuid)
        );
    }


    private function duplicateContent(Content $content, array $relatedContentTypes, string $tvScrennUuid): void
    {
        $data = array_except($content->getAttributes(), [
            'uuid', 'uuid_host','author','author_id', 'update_author', 'pid','created_at','updated_at','published_at'
        ]);
        $data['tv_screens'] = $tvScrennUuid;
        foreach ($relatedContentTypes as $relatedContentType) {
            $target = str($relatedContentType)->snake()->toString();
            $data[$target] =  $content->{$relatedContentType}->pluck('uuid')->toArray();
        }

        foreach (['video', 'image'] as $fileField) {
            /**
             * @var string $originalFilePath
             */
            $originalFilePath = $content->{$fileField};
            if (!empty($originalFilePath) && Storage::disk('local')->exists($originalFilePath)) {
                $data[$fileField] = $this->duplicateFile($originalFilePath);
            }
        }

        $bridge = new BridgeContent();
        $uuid = $bridge->contentInsert(
            $content->content_type,
            $data,
            true,
            true
        );
    }

    private function duplicateFile(string $originalFilePath): ?string
    {
        $isSuccess = false;
        $copyFile = '';
        if (!empty($originalFilePath) && Storage::disk('local')->exists($originalFilePath)) {
            $copyFile = sprintf(
                "%s/%s",
                dirname($originalFilePath),
                str(basename($originalFilePath))->prepend('1-')->toString()
            );
            $isSuccess =  Storage::disk('local')->copy($originalFilePath, $copyFile);
        }
        return $isSuccess ? $copyFile : null;
    }
}
