<?php

namespace Inside\TV\Console;

use Illuminate\Console\Command;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Settings\Models\Setting;

/**
 * Class InstallCommand
 *
 * @category Class
 * @package  Inside\TV\Console\InstallCommand
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class InstallCommand extends Command
{
    protected $name = 'inside:tv:install';

    /**
     * @var string
     */
    protected $signature = 'inside:tv:install';

    /**
     * @var string
     */
    protected $description = 'Install modules tv';

    private ?BridgeContentType $bridgeContentType = null;
    private array $styles = [];

    private const DEFAULT_DURATION_VIDEO = 8;
    private const INSIDE_TV_STYLE_IMAGE = 'image_inside_tv';

    /**
     * Handle index command
     *
     * @return void
     * @throws \Exception
     */
    public function handle(): void
    {
        $this->info('[Inside TV] : Start install contents for module TV');
        $this->line('');
        $this->createStyleImageTv();
        $this->createSettings();
        $this->loadStyles();
        $this->createTvContents();
        $this->updateCoreContents();

        $this->line('[Inside TV] : Finish install module\'s contents');
        $this->call('models:rebuild');
    }

    private function createStyleImageTv(): void
    {
        $bridgeContent = new BridgeContent();
        $langcode = config('app.locale', 'fr');

        $bridgeContent->contentInsert('image_styles', [
            'type' => 'node',
            'bundle' => 'image_styles',
            'langcode' => $langcode,
            'title' => self::INSIDE_TV_STYLE_IMAGE,
            'width' => 1920,
            'height' => 1080,
            'optimisation' => true,
            'optimisation_ratio' => 80,
            'resizable' => false
        ]);
    }

    private function createSettings(): void
    {
        Setting::firstOrCreate(['group' => 'tv', 'key' => 'screen_display_time'], ['value' => self::DEFAULT_DURATION_VIDEO]);
    }

    private function createTvContents(): void
    {
        $modules = config('tv.modules');
        foreach ($modules as $module) {
            $file = __DIR__ . '/../../database/contents/'.$module.'/content_type.php';
            if (file_exists($file)) {
                $this->line("[Inside TV] : Create or Update content $module");
                $content = include($file);
                $this->createOrUpdateModule($content);
                $this->appendStyleImages($content);
            }
        }
    }

    private function updateCoreContents(): void
    {
        foreach (['news', 'events'] as $contentType) {
            if (class_exists(type_to_class($contentType))) {
                $this->updateImageFieldOptions($contentType);
            }
        }
    }

    private function createOrUpdateModule(array $content): void
    {
        $contentType = $content['options']['name'];
        $this->getBridgeContentType()->contentTypeUpdateOrCreate(
            type: $contentType,
            options: $content['options'],
            fields: $content['fields']
        );
        // hide weight
        $this->getBridgeContentType()->changeFieldFormOptions($contentType, 'weight', false, 'group_options');
    }

    private function appendStyleImages(array $content): void
    {
        $contentType = $content['options']['name'];

        foreach ($content['fields'] as $field) {
            if ($field['name'] === 'image') {
                $this->updateImageFieldOptions($contentType);
            }
        }
    }

    private function updateImageFieldOptions(string $contentType): void
    {
        $this->getBridgeContentType()->updateFieldOptions(
            type: $contentType,
            fieldName: 'image',
            options: [
                'type' => BridgeContentType::IMAGE_FIELD,
                'special' => [
                    'image_styles' => $this->styles
                ]
            ]
        );
    }

    private function loadStyles(): void
    {
        $styles = ImageStyles::pluck('title')->all();
        if (!in_array(self::INSIDE_TV_STYLE_IMAGE, $styles)) {
            $styles[] = self::INSIDE_TV_STYLE_IMAGE;
        }
        $this->styles = collect($styles)->unique()->toArray();
    }

    private function getBridgeContentType(): BridgeContentType
    {
        return $this->bridgeContentType ??= new BridgeContentType();
    }
}
