<?php

use Illuminate\Database\Migrations\Migration;
use Inside\Content\Facades\ContentCache;
use Illuminate\Support\Facades\Artisan;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Permission\Events\BackofficeEntriesPermissionChangedEvents;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Services\PermissionSchemaService;
use Inside\Authentication\Models\User;
use Inside\Host\Bridge\BridgeContent;
use Inside\Facades\Package;
use Inside\Permission\Models\Role;
use Inside\TV\Services\TVContentFilterService;
use Inside\Content\Facades\Schema;

class AddTvUserAndRole extends Migration
{
    public function up(): void
    {
        // run install to set up new conf and update contents to make them permissible
        Artisan::call('inside:tv:install');

        $user = $this->getUserTv();
        /**
         * @var Role $role
         */
        $role = Role::firstOrCreate(['name' => config('tv.role_tv')]);
        $this->updatePermissions($role);
        $user->assignRole($role);
        $this->disableNotifications($user);
    }

    private function getUserTv(): User
    {
        $userUuid = User::where('email', config('tv.email_tv'))->first()?->uuid;

        if (blank($userUuid)) {
            $bridgeContent = new BridgeContent();
            $userAdmin = User::where('email', config('tv.email_admin'))->first();
            $attributes = array_merge(
                $userAdmin->information->getAttributes(),
                [
                    'email' => config('tv.email_tv'),
                    'name' => config('tv.email_tv'),
                    'firstname' => 'Gestion accès',
                    'lastname' => 'TV',
                    'status' => '1',
                    'is_maintenance' => true,
                    'is_gdpr_compliant' => true,
                    'langcode' => config('app.locale', 'fr')
                ]
            );

            $attributes = array_except(
                $attributes,
                [
                    'uuid',
                    'uuid_host',
                    'author',
                    'author_id',
                    'update_author',
                    'pid',
                    'created_at',
                    'updated_at'
                ]
            );
            $userUuid = $bridgeContent->contentInsert(
                type: 'users',
                data: $attributes,
                creation: true,
                fromCli: true
            );
        }

        /**
         * @var User $user
         */
        $user = User::where('uuid', $userUuid)->first();

        return $user;
    }

    private function updatePermissions(Role $role): void
    {
        $service = new TVContentFilterService();
        $validContentTypes = $service->getValidContentTypes();

        $schemas = $validContentTypes->map(fn ($type) => [
            'authorizable_type' => type_to_class($type),
            'action'            => 'read',
            'children'          => data_get(Schema::getModelOptions($type), 'categorizable'),
            'invert'            => false,
            'authorizable_uuid' => null,
        ])->toArray();

        PermissionSchema::create($schemas, $role->id);
        Permission::buildPermissionForRole($role->id);
        PermissionSchema::clean();
        ContentCache::forget([config('app.key', 'sid2'), 'all']);
    }

    private function disableNotifications(User $user): void
    {
        NotificationSubscriber::where('user_uuid', $user->uuid)->delete();
    }
}
