<?php

namespace Inside\Search\Solr\Engines;

use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Facades\ScopeLogic;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Facades\Package;
use Inside\Search\Facades\Searchable;
use Inside\Search\Services\Search;
use Inside\Search\Solr\Facades\Solr;
use Laravel\Scout\Builder;
use Laravel\Scout\Engines\Engine;
use Solarium\Exception\HttpException;
use Solarium\Plugin\BufferedAdd\BufferedAdd;
use Solarium\QueryType\Select\Query\Query;
use Solarium\QueryType\Select\Result\Document;
use Solarium\QueryType\Select\Result\Result;
use Solarium\QueryType\Select\Result\Result as SelectResult;

/**
 * SolR facade
 *
 * @category Class
 * @package  Inside\Search\Solr\Engines\SolrEngine
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class SolrEngine extends Engine
{
    /**
     * Perform the given search on the engine.
     *
     * @param Builder $builder
     *
     * @return mixed
     * @throws Exception
     */
    public function search(Builder $builder): mixed
    {
        return Solr::select($this->prepareQuery(Solr::createSelect(['querydefaultoperator' => 'AND']), $builder));
    }

    /**
     * Perform the given search on the engine.
     */
    public function paginate(Builder $builder, $perPage, $page): SelectResult
    {
        $offset = ($page - 1) * $perPage;
        $query = $this->prepareQuery(Solr::createSelect(), $builder, $offset, $perPage);
        return Solr::select($query);
    }

    /**
     * Get the total count from a raw result returned by the engine.
     *
     * @param Result $results
     *
     * @return int
     */
    public function getTotalCount($results): int
    {
        return (int)$results->getNumFound();
    }

    /**
     * Pluck and return the primary keys of the given results.
     *
     * @param Result $results
     *
     * @return Collection
     */
    public function mapIds($results): Collection
    {
        $ids = [];

        foreach ($results as $document) {
            $ids[] = $document->id;
        }

        return collect($ids);
    }

    /**
     * Map the given results to instances of the given model.
     *
     * @param Builder $builder
     * @param Result $results
     * @param mixed $model
     *
     * @return Collection
     */
    public function map(Builder $builder, $results, $model): Collection
    {
        if ($results->getNumFound() === 0) {
            return Collection::make();
        }

        // Trick to get request parameters
        Request::capture();
        $models = [];

        foreach ($results as $document) {
            $attributes = [];

            if (!isset($document['indexable_type'])) {
                continue; // Not a valid document
            }

            foreach ($document as $field => $value) {
                $attributes[$field] = $value;

                if ($field === 'id') {
                    $attributes['uuid'] = $value;
                }
            }

            // Dispatch to the correct model
            $modelClass = inside_search_get_class_from_filter(
                inside_search_solr_get_type_from_solr_type($attributes['indexable_type'])
            );

            if (!$modelClass) {
                continue;
            }

            // Get the real Model
            try {
                $entity = $modelClass::findOrFail($attributes['uuid']);

                // Adding search score
                $entity->score = $attributes['score'];
                // Adding slug
                if (isset($entity->slug)) {
                    $entity->url = count($entity->slug) > 0 ? $entity->slug[0] : '';
                }
                // Aggregate content types
                $indexable_type = $attributes['indexable_type'];

                // Adding type
                if (!isset($entity->type)) {
                    $entity->content_type = inside_search_solr_get_type_from_solr_type($indexable_type);
                }

                $models[] = $entity;
            } catch (ModelNotFoundException) {
                Log::error("[SolrEngine::map] failed to find model [$modelClass] of uuid <{$attributes['uuid']}>");
            }
        }

        return collect($models);
    }

    /**
     * Update the given model in the index.
     *
     * @param \Illuminate\Database\Eloquent\Collection $models
     *
     * @return void
     * @throws Exception
     */
    public function update($models): void
    {
        if ($models->isEmpty()) {
            return;
        }

        try {
            /**
             * @var BufferedAdd $buffer
             */
            $buffer = Solr::getPlugin('bufferedadd');
            $buffer->setBufferSize(50);

            $update = Solr::createUpdate();

            $documents = [];
            $models->each(function ($model) use (&$update, &$documents) {
                $searchable = $model->toSearchableArray();

                // Ensure the 'updated_at_date' field is present
                if (!isset($searchable['updated_at_date'])) {
                    $searchable['updated_at_date'] = $model->updated_at_date ?? now();
                }

                $document = $update->createDocument($searchable);
                $documents[] = $document;
            });

            $buffer->addDocuments($documents);
            $buffer->flush();
            $update->addCommit();

            $result = Solr::update($update);

            if ($result->getStatus() != 0) {
                throw new Exception("[Solr Engine] Update failed \n\n" . json_encode($result->getData()));
            }

            Log::debug(
                '[SolrEngine::update] update with status {' . $result->getStatus() . '} in (' . $result->getQueryTime() . ')'
            );
        } catch (Exception $e) {
            Log::error('[SolrEngine::update] Exception: ' . $e->getMessage());
        }
    }

    /**
     * Remove the given model from the index.
     *
     * @param \Illuminate\Database\Eloquent\Collection $models
     *
     * @return void
     * @throws Exception
     */
    public function delete($models): void
    {
        if ($models->isEmpty()) {
            return;
        }

        try {
            $update = Solr::createUpdate();
            $models->each(
                function ($model) use (&$update) {
                    $update->addDeleteById($model->uuid);
                }
            );
            $update->addCommit();
            $result = Solr::update($update);
            if ($result->getStatus() != 0) {
                throw new Exception("[Solr Engine] Delete failed \n\n" . json_encode($result->getData()));
            }
            Log::debug(
                '[SolrEngine::delete] delete with status {' . $result->getStatus() . '} in (' . $result->getQueryTime()
                . ')'
            );
        } catch (Exception $e) {
            Log::error('[SolrEngine::delete] failed to delete {' . $e->getMessage() . '}');
        }
    }

    /**
     * Prepare a term for  solr query
     *
     * @param string $term
     * @return string
     */
    private function escapeTerm(string $term): string
    {
        /**
         * From doc
         * Lucene supports escaping special characters that are part of the query syntax.
         * The current list special characters are
         * + - && || ! ( ) { } [ ] ^ " ~ * ? : \ /
         * To escape these characters use the \ before the character. For example to search for (1+1):2 use the query:
         * \(1\+1\)\:2
         */
        $pattern = '/( |\+|-|&&|\|\||!|\(|\)|\{|}|\[|]|\^|"|~|\*|\?|:|\/|\\\)/';

        return (string)preg_replace($pattern, '\\\$1', $term);
    }

    /**
     * Prépare a phrase for solr query
     *
     * @param string $phrase
     * @return string
     */
    private function escapePhrase(string $phrase): string
    {
        return '"' . preg_replace('/("|\\\)/', '\\\$1', trim($phrase)) . '"';
    }

    /**
     * Reformat query and set limit / offset
     *
     * @param  Query  $query  init query
     * @param  Builder  $builder  our builder
     * @param  int  $offset  optional offset, limit must be set to be used
     * @param  int|null  $limit  limit the query result
     *
     * @return Query prepared query
     */
    private function prepareQuery(Query $query, Builder $builder, int $offset = 0, int $limit = null): Query
    {
        $conditions = [];
        $term = $this->escapePhrase($builder->query);
        $terms = array_map(
            function ($item) {
                return $this->escapeTerm($item);
            },
            explode(' ', trim($builder->query))
        );

        $capture = Request::capture();
        $isAutocomplete = (bool)$capture->get('autocomplete', false);
        if (!$isAutocomplete) {
            $theSpellTerms = [];
            $theBodyTerms = [];
            foreach ($terms as $t) {
                $t = str_replace(':', '', $t);
                if (!empty($t)) {
                    $theSpellTerms[] = " spell:$t ";
                    $theBodyTerms[] = " body:$t ";
                }
            }
            if (!empty($theSpellTerms) && !empty($theBodyTerms)) {
                $titleSearchQuery = '';
                if ($indexableType = $builder?->wheres['indexable_type'] ?? false) {
                    $type = class_to_type($indexableType);
                    if ($type !== 'users') {
                        $contentOptions = Schema::getModelOptions($type);
                        $boost = array_key_exists('title_boost', $contentOptions) ? $contentOptions['title_boost'] : 2;
                        $titleSearchQuery = '(title:' . $this->escapeTerm($term) . '^' . $boost . ') OR ';
                    }
                }

                if (isset($builder->wheres['indexable_type IN'])) {
                    $titleSearchQuery = '(title:' . $this->escapeTerm($term) . '^4) OR ';
                }

                $conditions = !empty($term) ? [
                    "(".$titleSearchQuery."(spell:$term^2 OR (" . implode(' AND ', $theSpellTerms) . ")) OR (body:$term^2 OR (" . implode(
                        ' AND ',
                        $theBodyTerms
                    ) . ")))",
                ] : [];
            }
        }

        $importants = [];
        $searchableClasses = Searchable::getSearchableClasses();
        $indexedTypes = [];
        $isGlobal = false;
        foreach ($builder->wheres as $key => &$value) {
            if ($key == '_global') {
                $isGlobal = $value;
                continue;
            }

            $orSolrConditions = [];
            if (is_array($value) && $key == 'or') {
                foreach ($value as $orQueryFieldName => $orQuery) {
                    $orConditions = [];

                    foreach ($orQuery as $key => $value) {
                        if (strstr($key, 'created_at') || strstr($key, 'updated_at')) {
                            [$field, $value] = inside_search_solr_get_date_filter($key, $value);
                            $orConditions[] = "$field:$value";
                        } else {
                            if (is_array($value)) {
                                // TODO handle or condition with 2 different reference fields
                                // ((type_field_name_reference: ('a','b')) OR (type_other_field_name_reference: ('a','b')))
                                continue;
                            } else {
                                $orConditions[] = "$key:$value";
                            }
                        }
                    }

                    if (!empty($orConditions)) {
                        $orSolrConditions[] = '(' . implode(' AND ', $orConditions) . ')';
                    }
                }

                if ($orSolrConditions) {
                    $conditions[] = '(' . implode(' OR ', $orSolrConditions) . ')';
                }
            } elseif (is_array($value)) {
                if (in_array(null, $value)) {
                    $value[] = '--';
                    $value = array_filter($value);
                }
                $conditions[] = "$key:(\"" . implode("\" OR \"", $value) . "\")";
            } elseif (Str::endsWith($key, ' IN')) {
                $key = Str::substr($key, 0, -3);
                $value = explode('","', Str::substr(trim($value), 2, -2));
                if ($key === 'indexable_type') {
                    foreach ($value as $k => &$v) {
                        if ($searchableClasses->contains(
                            function ($searchable) use ($v) {
                                return $searchable->model === $v;
                            }
                        )
                        ) {
                            $v = app($v)->searchableAs();

                            if ($isAutocomplete) {
                                // Only search in important column
                                $type = inside_search_solr_get_type_from_solr_type($value[$k]);
                                $importantColumn = isset($searchableClasses[$type])
                                    ? $searchableClasses[$type]->options['searchable_important']
                                    : null;

                                if ($importantColumn) {
                                    $importants[$importantColumn] = $importantColumn;
                                }
                            }
                        } else {
                            unset($value[$k]);
                        }
                    }
                    $indexedTypes = array_filter($value, function ($value) {
                        return $value != 'inside_users_index';
                    });
                }
                $conditions[] = "$key:(\"" . implode("\" OR \"", $value) . "\")";
            } elseif ($key === 'indexable_type') {
                if ($searchableClasses->contains(
                    function ($searchable) use ($value) {
                        return ($searchable->model === $value);
                    }
                )
                ) {
                    $value = app($value)->searchableAs();
                    $indexedTypes = $value != 'inside_users_index' ? [$value] : [];
                    $conditions[] = "$key:$value";
                }
            } elseif (strstr($key, 'created_at') || strstr($key, 'updated_at') || strstr($key, 'published_at')) {
                [$field, $value] = inside_search_solr_get_date_filter($key, $value);
                $query->createFilterQuery($field)->setQuery($field . ':' . $value);
            } elseif ($key === 'archived') {
                if (Package::has('inside-archive')) {
                    if (is_bool($value)) {
                        $value = $value ? 'true' : 'false';
                    } elseif (is_string($value)) {
                        $value = '"' . addslashes(trim($value)) . '"';
                    }
                    $conditions[] = "archived_boolean:$value";
                }
            } else {
                if (is_bool($value)) {
                    $value = $value ? 'true' : 'false';
                } elseif (is_string($value)) {
                    $value = '"' . addslashes(trim($value)) . '"';
                }
                $conditions[] = "$key:$value";
            }
        }

        if (count($importants) > 0 && !empty($builder->query)) {
            $importants_or = [];
            $importants_strict_or = [];

            foreach ($importants as $imp) {
                if (!in_array($imp, ['title'])) {
                    $imp = "{$imp}_text";
                }
                $importants_or[] = $imp . ':' . $this->escapeTerm('*' . $builder->query . '*');
                $importants_strict_or[] = $imp . ':' . $this->escapeTerm($builder->query);
            }
            $conditions[] = "((spell:$term OR body:$term)  (\"" . implode("OR\"", $importants_or) . ")^2 (\"" .
                implode("OR\"", $importants_strict_or) . ")^10)";
        } else {
            if ($isAutocomplete && !empty($term)) {
                $conditions[] = "(spell:$term OR body:$term)";
            }
        }

        $customQueries = config('search.custom_queries', []);
        foreach ($customQueries as $customQuery) {
            if (is_callable($customQuery)) {
                $customQuery($query, $isGlobal);
            }
        }

        if ($isGlobal) {
            if (!empty($indexedTypes)) {
                $query->createFilterQuery('is_maintenance')
                    ->setQuery("is_maintenance_boolean:false OR indexable_type:(\"" .
                        implode('"OR"', $indexedTypes) . "\")");
            } else {
                $query->createFilterQuery('is_maintenance')->setQuery("is_maintenance_boolean:false");
            }
        }

        $user = Auth::user();
        $statusScope = $publishableScope = is_null($user);

        // Permission
        $wheresTypes = isset($builder->wheres['indexable_type IN']) ? collect($builder->wheres['indexable_type IN']) : collect([]);
        $whereType = $builder->wheres['indexable_type'] ?? null;
        if (!$user->hasRole('super_administrator') && !$wheresTypes->contains('inside_users_index') && $whereType !== 'inside_users_index') {
            $roleIds = $user->roles->pluck('id');
            $query->createFilterQuery('permission')->setQuery(
                "roles:(" . $roleIds->implode(' OR ') . ")"
            );
        }

        if (!$statusScope) {
            foreach ($indexedTypes as $type) {
                if (ScopeLogic::statusScopeNeeded($user, $type)) {
                    $statusScope = true;
                    break;
                }
            }
        }
        if (!$publishableScope) {
            foreach ($indexedTypes as $type) {
                if (ScopeLogic::publishabledScopeNeeded($user, $type)) {
                    $publishableScope = true;
                    break;
                }
            }
        }
        if ($statusScope) {
            $query->createFilterQuery('published_by_status')->setQuery('status:true');
        }
        if ($publishableScope) {
            $query->createFilterQuery('published_by_date')
                ->setQuery('published_at_date:[* TO ' . now()->toISOString() . ']');
        }

        Log::debug('[SolrEngine] Prepare query [' . implode(' AND ', $conditions) . ']');
        $query->setQuery(implode(' AND ', $conditions));

        if (!is_null($limit)) {
            $query->setStart($offset)->setRows($limit);
        }

        // Order
        $query->clearSorts();
        foreach ($builder->orders as $order) {
            $column = $order['column'];
            if (!in_array($column, ['status', 'title', 'timestamp', 'score',])) {
                if (in_array($column, ['created_at', 'updated_at'])) {
                    $column .= '_date';
                } else {
                    $column .= '_string';
                }
            }
            $query->addSort(
                $column,
                Str::lower($order['direction']) === 'asc' ? $query::SORT_ASC : $query::SORT_DESC
            );
        }
        return $query;
    }

    public function specialSearch(array $contentTypes, string $terms = '', array $filters = [], array $sorts = ['score:desc'], int $perPage = 10, int $page = 1, bool $withTrash = false): array
    {
        if (empty($contentTypes)) {
            return [];
        }

        // Create Solr query
        $query = Solr::createSelect();

        // Default fields
        $query->setFields(['id', 'score', 'indexable_type', 'language', 'url']);

        // Permission
        $user = Auth::user();
        if (!$user->hasRole('super_administrator')) {
            $roleIds = $user->roles->pluck('id');
            $query->createFilterQuery('permission')->setQuery(
                "roles:(" . $roleIds->implode(' OR ') . ")"
            );
        }

        $solrTypes = array_map(
            function ($type) {
                return 'inside_' . $type . '_index';
            },
            $contentTypes
        );
        $query->createFilterQuery('contentQuery')->setQuery(
            "indexable_type:(" . implode(' OR ', $solrTypes) . ")"
        );

        // Prepare facet
        $facetSet = $query->getFacetSet();
        $facetSet->createFacetField('type')->setField('indexable_type');

        // Prepare highlight
        $highlightingQuery = $query->getHighlighting();
        $highlightingQuery->setFields(['title']);
        $highlightingQuery->setSimplePrefix('<span>');
        $highlightingQuery->setSimplePostfix('</span>');
        $prepareQuery = [];

        $contentFieldsMapping = [];
        foreach ($contentTypes as $content) {
            $contentFieldQuery = ['spell:' . $this->escapePhrase($terms)];
            $searchableClasses = Schema::getFieldListing(
                $content,
                function ($field) {
                    return array_key_exists('searchable', $field['options']) && $field['options']['searchable'];
                }
            );
            $contentOptions = Schema::getModelOptions($content);
            $contentBoost = 1;
            if (array_key_exists('content_type_boost', $contentOptions)) {
                $contentBoost = (int)$contentOptions['content_type_boost'];
                if ($contentBoost <= 0) {
                    $contentBoost = 1;
                }
            }
            if (!array_key_exists($content, $contentFieldsMapping)) {
                $contentFieldsMapping[$content] = [
                    'image' => $contentOptions['search_image_field_result'] ?? null,
                    'main_taxo' => $contentOptions['search_main_taxo_field_result'] ?? null,
                    'secondary_taxo' => $contentOptions['search_secondary_taxo_field_result'] ?? null,
                    'date' => $contentOptions['search_date_field_result'] ?? null,
                    'title' => $contentOptions['title'],
                ];
            }
            foreach ($searchableClasses as $searchable) {
                $options = Schema::getFieldOptions($content, $searchable);
                $boost = 1;
                if ($searchable == 'title') {
                    $boost = array_key_exists('title_boost', $options) ? $options['title_boost'] : 2;
                } else {
                    if (array_key_exists('field_boost', $options)) {
                        $boost = (int)$options['field_boost'];
                        if ($boost <= 0) {
                            $boost = 1;
                        }
                    }
                }
                if ($options['type'] == 'section' && array_key_exists('target', $options)) {
                    foreach ($options['target'] as $sectionType) {
                        $searchableSectionClasses = Schema::getFieldListing(
                            $sectionType,
                            function ($field) {
                                return array_key_exists('searchable', $field['options'])
                                    && $field['options']['searchable'];
                            }
                        );
                        foreach ($searchableSectionClasses as $sectionSearchable) {
                            $sectionFieldOptions = Schema::getFieldOptions($sectionType, $sectionSearchable);
                            $sectionFieldBoost = 1;
                            if (array_key_exists('field_boost', $sectionFieldOptions)) {
                                $sectionFieldBoost = (int)$sectionFieldOptions['field_boost'];
                                if ($sectionFieldBoost <= 0) {
                                    $sectionFieldBoost = 1;
                                }
                            }

                            $contentFieldQuery[] =
                                '(section_content_' . $content . '_' . $sectionFieldOptions['type'] . '_text' . ':'
                                . $this->escapeTerm($terms) . ($boost != 1 ? '^' . $boost : '') . ')' . ($contentBoost
                                != 1 ? '^' . $sectionFieldBoost : '');
                        }
                    }
                    continue;
                }
                $contentFieldQuery[] =
                    '(' . $searchable . ':' . $this->escapeTerm($terms) . ($boost != 1 ? '^' . $boost : '') . ')'
                    . ($contentBoost != 1 ? '^' . $contentBoost : '');
            }
            $contentQuery = [
                'indexable_type:inside_' . $content . '_index',
                '(' . implode('||', $contentFieldQuery) . ')',
            ];
            $prepareQuery[] = implode(" && ", $contentQuery);

            $resultable = [];

            $fields = Schema::getFieldListing($content);
            foreach ($fields as $field) {
                $options = Schema::getFieldOptions($content, $field);
                $prefix = '';
                $suffix = '';
                if (!in_array($field, ['status', 'title', 'langcode'])
                    && array_key_exists(
                        'search_result_field',
                        $options
                    )
                    && $options['search_result_field']
                ) {
                    switch ($options['type']) {
                        case 'checkbox':
                        case 'boolean':
                            $suffix = '_boolean';
                            break;
                        case 'integer':
                            $suffix = '_int';
                            break;
                        case 'double':
                            $suffix = '_double';
                            break;
                        case 'file':
                            $suffix = '_text';
                            break;
                        case 'reference':
                            $prefix = $content . '_';
                            $suffix = '_*';
                            break;
                        case 'date':
                        case 'timestamp':
                            $suffix = '_date';
                            break;
                        case 'section':
                            $field = 'section_content_*_text';
                            break;
                        default:
                            $suffix = '_text';
                            break;
                    }
                }
                if (in_array($options['type'], ['section', 'text', 'wysiwyg',])) {
                    $highlightingQuery->addFields($prefix . $field . $suffix);
                }
                $resultable[] = $prefix . $field . $suffix;
            }

            $dateFields = ['created_at', 'updated_at', 'published_at'];

            foreach ($dateFields as $dateField) {
                if (in_array($dateField, $fields)) {
                    continue;
                }

                $resultable[] = $dateField . '_date';
            }

            $query->addFields(implode(',', $resultable));
        }

        foreach ($sorts as $sort) {
            [$column, $direction] = strpos($sort, ':') ?
                explode(':', $sort, 2) :
                [$sort, 'asc'];

            $query->addSort($column, $direction == 'asc' ? $query::SORT_ASC : $query::SORT_DESC);

            if ($column == 'score') {
                $query->addSort('timestamp', $query::SORT_DESC);
            }
        }

        // Pagination
        if ($perPage && $perPage > 0) {
            $query->setRows($perPage);
            $query->setStart(($page - 1) * $perPage);
        }

        // Filter query on langcode
        $query->createFilterQuery('language')->setQuery('langcode:("' . $user->langcode . '","--")');

        // Filter on status ( search engine don't get trashed )
        if (!$withTrash) {
            $query->createFilterQuery('status')->setQuery("status:true");
        }

        $queryString = '(' . implode(") || (", $prepareQuery) . ')';

        $query->setQuery($queryString);

        Log::debug("[SolrEngine::specialSearch] $queryString");

        // Spellcheck
        $query->getSpellcheck()
            ->setCount(5)
            ->setBuild(true)
            ->setCollate(true)
            ->setOnlyMorePopular(true)
            ->setDictionary('default');

        $resultSet = Solr::select($query);

        $spellcheckResults = $resultSet->getSpellcheck();

        $suggestions = [];
        if ($spellcheckResults && !$spellcheckResults->getCorrectlySpelled()) {
            foreach ($spellcheckResults as $spellcheckResult) {
                foreach ($spellcheckResult->getWords() as $word) {
                    $suggestions[$word['word']] = $word['freq'];
                }
            }
        }

        $result = [
            'total' => $resultSet->getNumFound(),
            'maxScore' => $resultSet->getMaxScore(),
            'types' => [],
            'data' => [],
            'spellcheck' => $suggestions,
        ];

        // Format facets results to filters
        $facet = $resultSet->getFacetSet()->getFacet('type');
        foreach ($facet as $value => $count) {
            $matches = [];
            if (!in_array($value, $solrTypes)) {
                continue;
            }
            if (preg_match("/^inside_(.+)_index$/", $value, $matches) !== false) {
                $result['types'][$matches[1]] = $count;
            }
        }

        // Suggester
        $suggesterQuery = Solr::createSuggester();
        $suggesterQuery->setDictionary(['insideSuggester', 'insideAiSuggester']);
        $suggesterQuery->setQuery($terms);
        //        $suggesterQuery->setBuild(true);
        $suggesterQuery->setCount(20);
        $suggesterQuery->setContextFilterQuery(implode(' ', $solrTypes));
        try {
            $suggestResultset = Solr::suggester($suggesterQuery);
        } catch (HttpException $exception) {
            // http://dev.intranet-inside.com:8983/solr/bclh/suggest?suggest=true&suggest.build=true&suggest.dictionary=insideSuggester&wt=jsonc
            $resetSuggesterQuery = Solr::createSuggester();
            $resetSuggesterQuery->setDictionary(['insideSuggester', 'insideAiSuggester']);
            //            $resetSuggesterQuery->setBuild(true);
            Solr::suggester($resetSuggesterQuery);
            $suggestResultset = Solr::suggester($suggesterQuery);
        }
        $result['suggests'] = [];
        foreach ($suggestResultset as $dictionary => $terms) {
            $result['suggests'][$dictionary] = [];
            foreach ($terms as $term => $termResults) {
                $result['suggests'][$dictionary][$term] = [];
                foreach ($termResults as $termResult) {
                    $result['suggests'][$dictionary][$term][] = $termResult['term'];
                }
            }
        }

        $highlightingQuery = $resultSet->getHighlighting();

        // Format search results
        foreach ($resultSet as $document) {
            $matches = [];
            if (preg_match("/^inside_(.+)_index$/", $document->indexable_type, $matches) !== false) {
                $type = $matches[1];
                $formattedResult = [
                    'image' => ($contentFieldsMapping[$type]['image'] !== null)
                        ? $document->{$contentFieldsMapping[$type]['image']} : $document->image,
                    'title' => $document->title,
                    'body' => '',
                    'main_taxo' => [],
                    'secondary_taxo' => [],
                    'date' => ($contentFieldsMapping[$type]['date'] !== null)
                        ? $document->{$contentFieldsMapping[$type]['date'] . '_date'} : $document->created_at_date,
                    'type' => $type,
                    'type_title' => $contentFieldsMapping[$type]['title'],
                    'url' => '',
                ];

                if ($contentFieldsMapping[$type]['main_taxo'] !== null) {
                    $formattedResult = $this->prepareTaxo(
                        $document,
                        $type,
                        $contentFieldsMapping[$type]['main_taxo'],
                        'main_taxo',
                        $contentFieldsMapping,
                        $formattedResult
                    );
                }
                if ($contentFieldsMapping[$type]['secondary_taxo'] !== null) {
                    $formattedResult = $this->prepareTaxo(
                        $document,
                        $type,
                        $contentFieldsMapping[$type]['secondary_taxo'],
                        'secondary_taxo',
                        $contentFieldsMapping,
                        $formattedResult
                    );
                }
                // Reformat date
                if ($formattedResult['date'] !== null) {
                    $formattedResult['date'] = Carbon::parse($formattedResult['date'])->timestamp;
                }
                $highlightedDoc = $highlightingQuery->getResult($document->id);

                foreach ($highlightedDoc as $field => $highlight) {
                    if (empty($highlight)) {
                        continue;
                    }
                    if ($field == 'title') {
                        $formattedResult['title'] = $highlight[0];
                    } elseif ($formattedResult['body'] === '') {
                        $formattedResult['body'] = $highlight[0];
                    }
                }

                /**
                 * $result['data'][$matches[1]][$document->id] = $document->getFields();
                 * $highlightedDoc                             = $highlighting->getResult($document->id);
                 * if ($highlightedDoc) {
                 * foreach ($highlightedDoc as $field => $highlight) {
                 * $result['data'][$matches[1]][$document->id]['highlight'][$field] = $highlight;
                 * }
                 * }
                 */
                $result['data'][$document->id] = $formattedResult;
            }
        }

        return $result;
    }

    protected function prepareTaxo(Document $document, string $type, string $target, string $taxoName, array $contentFieldsMapping, array $formattedResult): array
    {
        // Prepare main_taxo
        if (is_array($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_title_text'})) {
            foreach ($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_title_text'} as $key => $value) {
                $formattedResult[$taxoName][$key]['title'] = $value;
            }
        }
        if (is_array($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_reference'})) {
            foreach ($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_reference'} as $key => $value) {
                $formattedResult[$taxoName][$key]['uuid'] = $value;
            }
        }

        try {
            $imageFields = Schema::getFieldListingOfType($target, 'image');

            foreach ($imageFields as $imageField) {
                $options = Schema::getFieldOptions($target, $imageField);
                if (is_array($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_' .
                $imageField . '_main_text'})) {
                    foreach ($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_' .
                    $imageField . '_main_text'} as $key => $value) {
                        $formattedResult[$taxoName][$key][$imageField]['main'] = $value;
                    }
                }
                if (array_key_exists('image_styles', $options) && !empty($options['image_styles'])) {
                    $styles = ImageStyles::find($options['image_styles']);
                    foreach ($styles as $style) {
                        $attribute = $type . '_' . $contentFieldsMapping[$type][$taxoName] . '_' .
                            $imageField . '_' . Str::slug($style->title) . '_text';

                        if (!isset($document->{$attribute})) {
                            continue;
                        }

                        foreach ($document->{$attribute} as $key => $value) {
                            $formattedResult[$taxoName][$key][$imageField][Str::slug($style->title)] = $value;
                        }
                    }
                }
            }
        } catch (ModelSchemaNotFoundException $exception) {
            Log::error('[SolrEngine::prepareTaxo] Failed to prepare taxo => ' . $exception->getMessage());
        }

        if (is_array($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_color_text'})) {
            foreach ($document->{$type . '_' . $contentFieldsMapping[$type][$taxoName] . '_color_text'} as $key => $value) {
                $formattedResult[$taxoName][$key]['color'] = $value;
            }
        }

        return $formattedResult;
    }

    /**
     * Make an advanced search ( using filters/facets )
     *
     * @param array $contentTypes content type as type string
     * @param string $terms a string of terms separated by whitespaces
     * @param array $filters an array of filters ( should be tag as facet )
     * @param array $sorts our result sort
     * @param int $perPage result pagination
     * @param int $page result pagination
     * @param bool $withTrash does result get trashed item ?
     * @return array results
     */
    public function advancedSearch(
        array  $contentTypes,
        string $terms = '',
        array  $filters = [],
        array  $sorts = ['title:asc'],
        int    $perPage = 10,
        int    $page = 1,
        bool   $withTrash = false
    ): array {
        if (empty($contentTypes)) {
            return [];
        }

        // Prepare known facets
        $facets = null;
        foreach ($contentTypes as $content) {
            $fields = [];
            foreach (Schema::getFieldListing($content) as $field) {
                $options = Schema::getFieldOptions($content, $field);
                if ($options['filter_widget'] != '') {
                    $fields[$field] = $this->getFilterValue($filters, $field);
                }
            }
            if ($facets === null) {
                $facets = $fields;
            } else {
                $facets = array_intersect_assoc($facets, $fields);
            }
        }

        // Create Solr query
        $query = Solr::createSelect();

        // Prepare query
        if (!empty($terms)) {
            $modelOptions = Schema::getModelOptions(Arr::first($contentTypes));
            $importantColumn = $modelOptions['searchable_important'] ?? null;
            if (!empty($importantColumn)) {
                $importantColumn .= '_text';
                Log::debug("[SolrEngine] Prepare query [body:" . $this->escapePhrase($terms) . " " .
                    $importantColumn . ":" . $this->escapePhrase($terms) . "^2]");

                $query->setQuery("body:" . $this->escapePhrase($terms) . " " . $importantColumn . ":" .
                    $this->escapePhrase($terms) . "^2");
            } else {
                Log::debug("[SolrEngine] Prepare query [body:" . $this->escapePhrase($terms) . "]");
                $query->setQuery("body:" . $this->escapePhrase($terms));
            }
        }

        // Prepare Query
        /** @var array $contentTypes */
        if (count($contentTypes) > 0) {
            // Filter query on content type
            $query->createFilterQuery('type')->setQuery(
                "indexable_type:(inside_" . implode("_index OR inside_", $contentTypes) . "_index)"
            );
        }

        // Fields
        $query->setFields(['id', 'indexable_type', 'score']);

        // Order
        $columnMapping = [
          'title' => 'title_sort',
          'published_at' => 'published_at_date'
        ];
        foreach ($sorts as $sort) {
            [$column, $direction] = strpos($sort, ':') ? explode(':', $sort, 2) : [$sort, 'asc'];

            if (isset($columnMapping[$column])) {
                $column = $columnMapping[$column];
            } else {
                try {
                    $fieldOptions = Schema::getFieldOptions(Arr::first($contentTypes), $column);
                    switch ($fieldOptions['type']) {
                        case 'checkbox':
                        case 'boolean':
                            $suffix = '_boolean';
                            break;
                        case 'integer':
                            $suffix = '_int';
                            break;
                        case 'double':
                            $suffix = '_double';
                            break;
                        case 'date':
                        case 'timestamp':
                            $suffix = '_date';
                            break;
                        default:
                            $suffix = '_text';
                            break;
                    }
                    $column .= $suffix;
                } catch (Exception $e) {
                    continue; // Wrong column
                }
            }

            $query->addSort($column, $direction == 'asc' ? $query::SORT_ASC : $query::SORT_DESC);

            if ($column == 'score') {
                // Second sort
                $query->addSort('timestamp', $query::SORT_DESC);
            }
        }

        // Pagination
        if ($perPage && $perPage > 0) {
            $query->setRows($perPage);
            $query->setStart(($page - 1) * $perPage);
        }

        // Filter query on langcode
        $user = Auth::user();
        $query->createFilterQuery('language')->setQuery('langcode:("' . $user->langcode . '","--")');

        // Filter on status ( search engine don't get trashed )
        if (!$withTrash) {
            $query->createFilterQuery('status')->setQuery("status:true");
        }

        $roleQuery = $user->roles->pluck('id')->map(function ($roleId) {
            return 'roles:' . $roleId;
        })->implode(' OR ');
        $query->createFilterQuery('permission')->setQuery($roleQuery);

        // Prepare facet set
        $facetSet = $query->getFacetSet();

        foreach ($facets as $key => $value) {
            // First content type is used as a "master"
            $options = Schema::getFieldOptions($contentTypes[0], $key);
            switch ($options['type']) {
                case 'reference':
                    if ($value !== null) {
                        // No value, let's query filter this instead of adding a facet field
                        if (is_array($value)) {
                            $operator = ' OR ';
                            if (isset($options['filter_query_mode']) &&
                                $options['filter_query_mode'] == Search::OPERATOR_AND) {
                                $operator = ' AND ';
                            }
                            $value = "(" . implode($operator, $value) . ")";
                        }
                        $query->createFilterQuery($key)->setQuery(
                            $contentTypes[0] . "_" . $key . "_reference:" . $value
                        );
                    }
                    $facetSet->createFacetField($key)->setField($contentTypes[0] . "_" . $key . "_reference")->setLimit(500);

                    break;
                case 'checkbox':
                case 'boolean':
                    if ($value !== null) {
                        $query->createFilterQuery($key)->setQuery(
                            $contentTypes[0] . "_" . $key . "_boolean:" . ($value ? "true" : "false")
                        );
                    }
                    $facetSet->createFacetField($key)->setField($key . "_boolean")->setLimit(200);

                    break;
            }
        }
        // Don't get values with no potential result
        if (env('ENABLE_INSIDE_ADVANCED_SEARCH_FUNNEL_SEARCH', false)) {
            $facetSet->setMinCount(1);
        }

        // Let's select
        $resultSet = Solr::select($query);

        // Prepare our result
        $result = [
            'total' => $resultSet->getNumFound(),
            'maxScore' => $resultSet->getMaxScore(),
            'filters' => [],
            'data' => [],
        ];

        $hideCatalogueFilterRules = config('catalogue.filters.hidden_rules', []);

        // Format facets results to filters
        foreach ($facets as $key => $value) {
            $result['filters'][$key] = [];
            $facet = $resultSet->getFacetSet()->getFacet($key);

            foreach ($hideCatalogueFilterRules as $callback) {
                if ($callback && is_callable($callback)) {
                    if (true === $callback($key, collect($filters)->keyBy('filter'))) {
                        continue 2; // Skip this filter
                    }
                }
            }

            foreach ($facet as $value => $count) {
                $result['filters'][$key][$value] = $count;
            }
        }

        // Format search results
        foreach ($resultSet as $document) {
            $matches = [];
            if (preg_match("/^inside_(.+)_index$/", $document->indexable_type, $matches) === 1) {
                $result['data'][$matches[1]][$document->id] = $document->score;
            }
        }

        return $result;
    }

    /**
     * Get values for field filter
     *
     * Note each filter should be unique
     *
     * @param array $filters
     * @param string $field
     *
     * @return array|null
     */
    protected function getFilterValue(array $filters, string $field): ?array
    {
        foreach ($filters as $filter) {
            if (($filter['filter'] == $field) && ($filter['operator'] == 'in')) {
                return $filter['values'];
            }
        }

        return null;
    }

    public function flush($model)
    {
        // TODO: method that should flush all records
    }
}
