<?php

namespace Inside\Search\Solr\Models\Traits;

use Exception;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Content\Models\Field;
use Inside\Facades\Package;
use Laravel\Scout\Searchable as ScoutSearchable;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use Vaites\ApacheTika\Client;

trait Searchable
{
    use ScoutSearchable;

    protected $titleReplacements = ['name', 'email'];

    protected $attributeFields = ['title', 'langcode', 'status', 'indexable_type', 'url', 'uuid'];

    /**
     * Get the indexable data array for the model.
     *
     * @return array
     * @throws Exception
     */
    public function toSearchableArray()
    {
        $attributes = collect();
        $fields = Field::whereHas(
            'model',
            function ($query) {
                $query->whereClass(get_class($this));
            }
        )->get();
        $contentType = class_to_type($this);

        foreach ($fields as $field) {
            $value = $this->attributes[$field->name] ?? null;
            if (!in_array($field->name, $this->attributeFields)) {
                $type = $field->type;
                $options = $field->options;
                $searchable = isset($options['searchable']) && $options['searchable'];
                $filterable = isset($options['searchable_filter']) && $options['searchable_filter'];
                $searchResult = isset($options['search_result_field']) && $options['search_result_field'];

                if (!$searchable && !$filterable && !$searchResult) {
                    continue;
                }

                if (($type != 'reference') && ($value === null)) {
                    continue;
                } // avoid indexing null stuff

                if (!$attributes->has('content')) {
                    $attributes['content'] = collect();
                }

                switch ($type) {
                    case 'checkbox':
                    case 'boolean':
                        $attributes[$field->name.'_boolean'] = (bool) $value;
                        break;
                    case 'integer':
                        $attributes[$field->name.'_int'] = (int) $value;
                        $attributes[$field->name.'_text'] = (string) $value;
                        break;
                    case 'double':
                        $attributes[$field->name.'_double'] = (float) $value;
                        $attributes[$field->name.'_text'] = (string) $value;
                        break;
                    case 'file':
                        $attributes[$field->name.'_text'] = $this->cleanText((string) $value." ".$this->getFileContent($value));
                        break;
                    case 'date':
                    case 'timestamp':
                        $attributes[$field->name.'_date'] =
                            Carbon::createFromFormat('Y-m-d H:i:s', $value)->toISOString();
                        break;
                    case 'reference':
                        $cardinality = $options['cardinality'] ?? -1;
                        $target = $options['target'] ?? [];

                        if ($cardinality === 1) {
                            // TODO: rework
                            $references =
                                $field->name != 'author' ? $this->{Str::camel($field->name)} : [$this->authors];
                        } else {
                            $references = $this->{Str::camel($field->name)};
                        }

                        $i = 0;
                        foreach ($references as $reference) {
                            if (!$attributes->has($contentType.'_'.$field->name.'_reference')) {
                                $attributes[$contentType.'_'.$field->name.'_reference'] = collect();
                            }
                            // *_reference field is multiValued and designed to index array of uuids
                            $attributes[$contentType.'_'.$field->name.'_reference'][] = (string) $reference->uuid;

                            if (isset($reference->title) && !empty($reference->title)) {
                                $attributes[$contentType.'_'.$field->name.'_'.$i.'_title_text'] =
                                    $this->cleanText($reference->title);
                                $attributes[$contentType.'_'.$field->name.'_'.$i++.'_title_string'] =
                                    $this->cleanText($reference->title);

                                if (!$attributes->has($contentType.'_'.$field->name.'_title_text')) {
                                    $attributes[$contentType.'_'.$field->name.'_title_text'] = collect();
                                }
                                $attributes[$contentType.'_'.$field->name.'_title_text'][] =
                                    $this->cleanText($reference->title);
                            } elseif ($field->name == 'author') {
                                // Author
                                if (!$attributes->has($contentType.'_'.$field->name.'_title_text')) {
                                    $attributes[$contentType.'_'.$field->name.'_title_text'] = collect();
                                }
                                $attributes[$contentType.'_'.$field->name.'_title_text'][] =
                                    ($reference->firstname ? Str::ucfirst(
                                        Str::lower($this->cleanText($reference->firstname))
                                    ).' ' : '').Str::upper($this->cleanText($reference->lastname));
                            }

                            // Try to index image & color
                            if (!empty($target)) {
                                $imageFields = Schema::getFieldListingOfType($target[0], 'image');
                                foreach ($imageFields as $imageField) {
                                    $fieldOptions = Schema::getFieldOptions($target[0], $imageField);
                                    if ($fieldOptions['searchable'] && $reference->{$imageField}) {
                                        $mainImageKey =
                                            $contentType.'_'.$field->name.'_'.$imageField.'_main_text';
                                        if (!$attributes->has($mainImageKey)) {
                                            $attributes[$mainImageKey] = collect();
                                        }
                                        $attributes[$mainImageKey][] = protected_file_url($reference, $imageField);
                                        $imageOptions =
                                            Schema::getFieldOptions(class_to_type($reference), $imageField);
                                        if (array_key_exists('image_styles', $imageOptions)
                                            && !empty($imageOptions['image_styles'])
                                        ) {
                                            $styles = ImageStyles::find($imageOptions['image_styles']);
                                            foreach ($styles as $style) {
                                                $styleImageKey =
                                                    $contentType.'_'.$field->name.'_'.$imageField.'_'
                                                    .Str::slug($style->title).'_text';
                                                if (!$attributes->has($styleImageKey)) {
                                                    $attributes[$styleImageKey] = collect();
                                                }
                                                $attributes[$styleImageKey][] =
                                                    protected_file_url($reference, $imageField, true, $style->title);
                                            }
                                        }
                                    }
                                }
                            }

                            if ($reference->color) {
                                $key = $contentType.'_'.$field->name.'_color_text';
                                $attributes[$key] = $reference->color;
                            }
                        }
                        break;
                    case 'text':
                        switch ($field->options['widget']) {
                            case 'phone':
                                if (is_string($value) && strlen($value) > 0) {
                                    $value = preg_replace('/[^\d+]/', '', $value);
                                    $phoneUtil = PhoneNumberUtil::getInstance();
                                    try {
                                        $number = $phoneUtil->parse($value, 'FR');

                                        $attributes[$field->name.'_e164_phone_text'] =
                                        $phoneE164 = $phoneUtil->format($number, PhoneNumberFormat::E164);
                                        $attributes[$field->name.'_national_phone_text'] = str_replace(
                                            ' ',
                                            '',
                                            $phoneUtil->format($number, PhoneNumberFormat::NATIONAL)
                                        );
                                        $shortNumberLength =
                                            $options['short_number_length'] ?? 4;
                                        if ($shortNumberLength > 0 && strlen($phoneE164) >= $shortNumberLength) {
                                            $attributes[$field->name.'_short_phone_text'] =
                                                substr($phoneE164, -$shortNumberLength);
                                        }
                                    } catch (NumberParseException $e) {
                                        Log::error(
                                            '[Searchable] phone number failed to be parsed {'.$value.'} => '
                                            .$e->getMessage()
                                        );
                                    }
                                }
                                break;
                        }
                        // Special text widgets adds some attributes, fallback to default
                        // no break
                    default:
                        $attributes[$field->name.'_text'] = $this->cleanText((string) $value);
                        $attributes[$field->name.'_string'] = $this->cleanText((string) $value);
                        $attributes['content'][] = $this->cleanText((string) $value);
                        break;
                }
                continue;
            }
            if ($value !== null) {
                $attributes[$field->name] = $this->cleanText($value);
            }
            // Add roles ( TODO: move this to permission )
            $attributes['roles'] = DB::table('inside_permissions')->where($this->getKeyName(), $this->getKey())->where(
                'type',
                get_class($this)
            )->where('action', 'read')->pluck('role_id');
        }

        // Archives
        if (Package::has('inside-archive')) {
            $service = new \Inside\Archive\Services\ArchiveService();
            $attributes['archived_boolean'] = (bool) $service->isArchived(get_class($this), $this->uuid);
        }

        if (Package::has('inside-workflow')) {
            $service = new \Inside\Workflow\Services\ProposalService();
            $proposal = $service->getFromContent($this->uuid, get_class($this));

            if ($proposal) {
                $attributes['workflow_status_int'] = (int) $proposal->status;
            }
        }

        // Special user case
        if (isset($this->attributes['firstname']) && isset($this->attributes['lastname'])
            && !empty($this->attributes['firstname'])
            && !empty($this->attributes['lastname'])
        ) {
            $attributes['fullname_text'] = trim(
                $this->attributes['firstname'].' '.$this->attributes['lastname']
            );
            $attributes['reversefullname_text'] = trim(
                $this->attributes['lastname'].' '.$this->attributes['firstname']
            );
            $attributes['title'] = $attributes['fullname_text'];
        }

        // Get sections
        $i = 1;
        if ($this->sectionContent && $this->sectionContent->isNotEmpty()) {
            foreach ($this->sectionContent as $content) {
                $fields = Field::whereHas(
                    'model',
                    function ($query) use ($content) {
                        $query->whereClass(get_class($content));
                    }
                )->get();

                if ($fields->count()) {
                    foreach ($fields as $field) {
                        if (isset($field->options['searchable']) && $field->options['searchable']) {
                            $value = $content->{$field->name};

                            if ($field->type == 'image') {
                                if ($content->{$field->name}) {
                                    $mainImageKey = 'section_content_'.class_to_type($content).'_'.$field->name
                                        .'_main_text';
                                    if (!$attributes->has($mainImageKey)) {
                                        $attributes[$mainImageKey] = collect();
                                    }
                                    $attributes[$mainImageKey][] = protected_file_url($content, $field->name);
                                    $imageOptions =
                                        Schema::getFieldOptions(class_to_type($content), $field->name);
                                    if (array_key_exists('image_styles', $imageOptions)
                                        && !empty($imageOptions['image_styles'])
                                    ) {
                                        $styles = ImageStyles::find($imageOptions['image_styles']);
                                        foreach ($styles as $style) {
                                            $styleImageKey =
                                                'section_content_'.class_to_type($content).'_'.$field->name.'_'
                                                .Str::slug($style->title).'_text';
                                            if (!$attributes->has($styleImageKey)) {
                                                $attributes[$styleImageKey] = collect();
                                            }
                                            $attributes[$styleImageKey][] =
                                                protected_file_url($content, $imageField, true, $style->title);
                                        }
                                    }
                                }
                            } elseif ($field->type == 'file') {
                                try {
                                    if (!empty($value)) {
                                        $filepath = $value;
                                        $value = $this->getFileContent($value);
                                        if ($value === null) {
                                            continue;
                                        } // avoid indexing null stuff
                                        $value .= " ".$filepath;
                                        $attributes['section_content_'.$i++.'_text'] =
                                            $this->cleanText(basename($content->file));
                                        if (isset($content->title) && !empty($content->title)) {
                                            $attributes['section_content_'.$i++.'_text'] =
                                                $this->cleanText($content->title);
                                        }
                                        if (isset($content->description) && !empty($content->description)) {
                                            $attributes['section_content_'.$i++.'_text'] =
                                                $this->cleanText($content->description);
                                        }
                                    }
                                } catch (Exception $e) {
                                    Log::error("Can't get file content [$value] {".$e->getMessage()."}");
                                }
                            }
                            if (!$attributes->has('section_content_'.$field->type.'_text')) {
                                $attributes['section_content_'.$field->type.'_text'] = collect();
                            }
                            $value = $this->cleanText($value);
                            if (!empty($value)) {
                                $attributes['section_content_'.$field->type.'_text'][] = $value;
                                $attributes['section_content_'.$i++.'_text'] = $value;

                                $attributes['content'][] = $value;
                            }
                        }
                    }
                }
            }
        }

        // try to get a title if we don't
        $replacements = env('INSIDE_SOLR_SEARCH_TITLE_REPLACEMENTS', null);
        if ($replacements === null) {
            $replacements = $this->titleReplacements;
        } else {
            $replacements = explode(',', $replacements);
        }

        while (!isset($attributes['title']) && count($replacements) > 0) {
            $replacement = array_pop($replacements);

            if (array_key_exists($replacement, $this->attributes)) {
                $attributes['title'] = $this->attributes[$replacement];
            }
        }

        $attributes['id'] = $this->uuid;
        $attributes['indexable_type'] = $this->searchableAs();
        $attributes['langcode'] =
            ($this->isTranslatable && $contentType !== 'users') ? $this->langcode : '--';
        $attributes['status'] = (bool) $this->status;
        $dateField = 'created_at';

        if (isset($options['search_date_field_result'])) {
            $dateField = $options['search_date_field_result'];
        }

        $attributes['timestamp'] = $this->getDateValue($dateField);
        $attributes['url'] = $this->slug ? $this->slug[0] : '';

        if (Schema::isContentType($contentType) && !in_array($contentType, ['comments'])) {
            $attributes['created_at_date'] = $this->getDateValue('created_at');
            $attributes['published_at_date'] = $contentType === 'users' ?
                $this->getDateValue('created_at') : $this->getDateValue('published_at');
            $attributes['updated_at_date'] = $this->getDateValue('updated_at');
        }

        return $attributes->toArray();
    }

    /**
     *
     * @return string
     */
    public function searchableAs()
    {
        $index = str_replace('\\', '', Str::snake(class_basename($this)));

        return 'inside_'.$index.'_index';
    }

    /**
     * clean text
     *
     * @param  string|null  $text
     * @return string
     */
    protected function cleanText(?string $text): string
    {
        if ($text == null) {
            return '';
        }

        return strip_tags(
            iconv("UTF-8", "UTF-8//IGNORE", preg_replace('/[\x00-\x09\x0B\x0C\x0E-\x1F\x7F]/', '', $text))
        );
    }

    /**
     * Using Tika to get File content
     *
     * @param  string  $path
     *
     * @return string
     * @throws Exception
     */
    protected function getFileContent(string $path): string
    {
        if (empty($path)) {
            return '';
        }

        try {
            setlocale(LC_ALL, "");
            $client = Client::make(config('tika.host'));
            if (!Storage::disk('local')->exists($path)) {
                Log::warning('File path ['.$path.'] do not exist');

                return '';
            }
            $realpath = Storage::disk('local')->path($path);

            // Convert path using local machine separators & get content
            $text = $client->getText(str_replace('/', DIRECTORY_SEPARATOR, $realpath));
        } catch (Exception $e) {
            Log::warning('Failed to load file content ['.$path.'] with Tika ['.$e->getMessage().']');

            return '';
        }

        return $text;
    }

    /**
     * return Date ISO Forrmat
     * @param string $dateField
     * @return string
     */
    protected function getDateValue(string $dateField): string
    {
        return $this->{$dateField} instanceof Carbon ?
            $this->{$dateField}->toISOString() :
            Carbon::createFromTimestamp($this->{$dateField})->toISOString();
    }
}
