<?php

namespace Inside\Search\Database\Services;

use Illuminate\Database\Connection;
use Illuminate\Database\DatabaseManager;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Inside\Support\Str;
use Inside\Content\Facades\ScopeLogic;
use Inside\Content\Models\Contents\Users;
use Inside\Search\Database\Models\Index;

/**
 * Searcher
 *
 * @category Class
 * @package  Inside\Search\Database\Services
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class SearcherService
{
    protected IndexerService $indexerService;

    /**
     * DB connection
     */
    protected Connection $connection;

    /**
     * SearcherService constructor.
     *
     * @param IndexerService  $indexerService
     * @param DatabaseManager $db
     */
    public function __construct(IndexerService $indexerService, DatabaseManager $db)
    {
        $this->indexerService = $indexerService;
        $this->connection = $db->connection();
    }

    /**
     * Search term using options
     *
     * @param string $term
     * @param array $options
     *
     * @return array
     */
    public function search(string $term, array $options = []): array
    {
        $searchIndex = new Index();
        $morphType = $searchIndex->indexable()->getMorphType();
        $foreignKey = $searchIndex->indexable()->getForeignKeyName();
        $isGlobal = false;

        // 1. Initialisation de la requête
        $query = Index::select($foreignKey, $morphType);

        // ... (Logique de filtres, gestion de rôles, etc. – inchangée) ...

        // Trick to get request parameters
        $capture = Request::capture();
        $importantSearch = (bool)$capture->get('autocomplete', false);
        $strictSearch = (bool)$capture->get('strict', false);

        if (Arr::get($options, 'index', '*') !== '*') {
            $query->where($morphType, $options['index']);
        }

        foreach (Arr::get($options, 'filters', []) as $filter) {
            if ($filter->getExpression() == '_global') {
                $isGlobal = $filter->getValue();
                continue;
            }

            $filter->apply($searchIndex->getTable(), $query);
        }

        /**
         * filter maintenance user in global search
         */
        if ($isGlobal) {
            $query->where('filter', 'like', '%status:1%');
            $query->where(
                function ($query) {
                    $query->where('indexable_type', '<>', Users::class)->orWhere(
                        'filter',
                        'not like',
                        '%is_maintenance:1%'
                    );
                }
            );
        }

        $specialRoles = array_unique(array_merge(config('permission.special_roles', []), ['super_administrator']));
        $user         = Auth::user();
        if ($user === null || !$user->permission->hasAnyRole($specialRoles)) {
            // Non special roles may not search unpublished contents
            $query->wherePublished(true);
            $query->where(function ($query) {
                $query->where('published_at', '<=', now())
                    ->orWhereNull('published_at');
            });
        }

        $customQueries = config('search.custom_queries', []);
        foreach ($customQueries as $customQuery) {
            if (is_callable($customQuery)) {
                $customQuery($query, $isGlobal);
            }
        }

        // Spit words
        $search = Str::lower(trim($term));
        preg_match_all('/(?:")((?:\\\\.|[^\\\\"])*)(?:")|(\S+)/', $search, $matches);
        $words = $matches[1];

        for ($i = 2; $i < count($matches); $i++) {
            $words = array_filter($words) + $matches[$i];
        }

        $stopwords = config('stopwords.words', []);
        $toSearch  = [];
        foreach ($words as $word) {
            if (strlen($word) >= config('scout.mysql.min_search_length', 3) && !in_array($word, $stopwords)) {
                $toSearch[] = $word;
            }
        }

        if (count($toSearch) == 0) {
            // Nothing to be searched !
            return [
                'count'   => 0,
                'results' => [],
            ];
        }

        // 2. Détermination du Poids Global (Rank)
        $rank = 10; // Poids par défaut pour les colonnes combinées
        if ($strictSearch) {
            $rank = 20; // Double le poids si recherche stricte
        } elseif ($importantSearch) {
            $rank = 15; // Augmente le poids si recherche importante
        }

        // 3. Génération de la clause de scoring FULLTEXT unique
        // On cible le nouvel index composite 'important_content,content'
        $queries = $this->getSearchQueries('important_content,content', $rank, $toSearch);

        // On ajoute une clause de recherche par titre
        $titleRank = $strictSearch ? 20 : ($importantSearch ? 15 : 10);
        $titleLikeQuery = $this->getTitleLikeQuery($toSearch, $titleRank);
        if (!empty($titleLikeQuery)) {
            $queries[] = $titleLikeQuery;
        }

        $selects = [];
        foreach ($queries as $select) {
            if (!empty($select)) {
                $selects[] = $select;
            }
        }

        // 4. Application du score dans la requête
        $query->selectRaw('( MAX(' . implode(' + ', $selects) . ') ' . ') as score');

        $minimalScore = $isGlobal
            ? config('scout.minimal_global_score', 25.0)
            : config('scout.minimal_score', 1.0);

        $weightCount = number_format($minimalScore, 2, '.', ''); // Score minimum de 1.0 (ou le minimum que vous souhaitez)
        $query->havingRaw('( max(' . implode(' + ', $selects) . ') ' . ') >= ' . $weightCount);

        // ... (Logique de GROUP BY et ORDER BY – inchangée) ...

        $groupRules = [$morphType, $foreignKey, $searchIndex->getTable().'.'.'title'];
        $query->groupBy($groupRules);

        $results['count'] = count($query->get());

        if (isset($options['limit'])) {
            $query->limit($options['limit']);
        }

        if (Arr::get($options, 'offset', 0) > 0) {
            $query = $query->offset($options['offset']);
        }

        if (isset($options['orders']) && is_array($options['orders'])) {
            if (isset($options['filters'][$morphType]) || isset($options['filters'][$morphType . ' IN'])) {
                /** @var \Inside\Search\Database\Models\FilterItem $item */
                $item = $options['filters'][$morphType] ?? $options['filters'][$morphType . ' IN'];
                $value = $item->getValue();
                $classes = [];
                if ($item->getOperator() == '=') {
                    $classes = [$value];
                } elseif ($item->getOperator() == 'in') {
                    $classes = $value;
                }

                foreach ($options['orders'] as $by => $direction) {
                    if (in_array($by, ['score', 'title', 'status', 'created_at', 'updated_at'])) {
                        $query->orderBy($by, $direction);
                        if ($by !== 'score') {
                            $groupRules[] = $by;
                        }

                        continue;
                    }

                    /** @var array $classes */
                    foreach ($classes as $class) {
                        $table = class_to_table($class);
                        $oldBy = $by;
                        if ($by == 'published_at' && $table == 'inside_content_users') {
                            $by = 'updated_at';
                        }
                        $query->leftJoin($table, $searchIndex->getTable() . '.' . $foreignKey, $table . '.uuid');
                        $query->orderBy($table . '.' . $by, $direction);
                        $groupRules[] = $table . '.' . $by;
                        $by = $oldBy;
                    }
                }
            } else {
                $query->orderBy('score', 'desc');
            }
        }

        $query->groupBy($groupRules);

        $results['results'] = $query->get()->map(
            function ($row) use ($morphType, $foreignKey) {
                return ['id' => $row->{$foreignKey}, 'type' => $row->{$morphType}, 'score' => $row->score];
            }
        );

        return $results;
    }

    protected function getTitleLikeQuery(array $words, int $rank): string
    {
        return collect($words)
            ->map(function ($word) use ($rank) {
                $word = addslashes($word);

                return "(
                CASE
                    WHEN LOWER(title) LIKE '%$word%'
                    THEN $rank
                    ELSE 0
                END
            )";
            })
            ->implode(' + ');
    }

    /**
     * Get search queries as weight
     *
     * // If found get points
     *
     * @param string $column
     * @param int $weight
     * @param array $words
     *
     * @return array
     */
    protected function getSearchQueries(string $column, int $weight, array $words): array
    {
        $queries = [];

        $queries[] = $this->getMatchSearchQuery('important_content,content', $words, $weight);

        return $queries;
    }

    /**
     * Get One search query using LIKE
     *
     * @param         $column
     * @param array   $words
     * @param         $rank
     * @param string  $prefix
     * @param string  $suffix
     *
     * @return string
     */
    protected function getSearchQuery(string $column, array $words, int $rank, string $prefix = '', string $suffix = ''): string
    {
        return collect($words)
            ->map(fn ($word) => strtolower(Str::wrap($word, $prefix, $suffix)))
            ->map(fn ($word) => $this->indexerService->getFullText()->buildLikeQuery($column, $word, $rank))
            ->implode(' + ');
    }

    /**
     * Get One search query using MATCH AGAINST
     *
     * @param string $columns
     * @param array $words
     * @param int $rank
     *
     * @return string
     */
    protected function getMatchSearchQuery(string $columns, array $words, int $rank): string
    {
        $term = implode(' ', $words);

        // 1. Chaîne de score (Pondération)
        $weightString = "\"$term\"^15";
        foreach ($words as $word) {
            // On nettoie le mot pour le FULLTEXT (enlever les : , ; etc)
            $cleanWord = preg_replace('/[[:punct:]]/u', '', $word);
            if (strlen($cleanWord) < 2) {
                continue;
            }

            $weightString .= " $cleanWord*^5 $cleanWord^1";
        }
        $weightString .= " WEIGHT(important_content, 3) WEIGHT(content, 1)";

        // 2. Chaîne de validation (Le Kill-Switch)
        $wildcardTerms = collect($words)->map(function ($w) {
            // Nettoyage strict pour le filtre
            $w = preg_replace('/[[:punct:]]/u', '', $w);
            if (strlen($w) < 2) {
                return null;
            }

            // Singulier/Pluriel : on réduit d'une lettre si mot long
            $root = (strlen($w) > 5) ? substr($w, 0, -1) : $w;
            return '+' . $root . '*';
        })->filter()->implode(' ');

        $searchClause = "MATCH($columns) AGAINST(LOWER('$weightString') IN BOOLEAN MODE)";
        $filterClause = "MATCH($columns) AGAINST('$wildcardTerms' IN BOOLEAN MODE)";

        // 3. Sécurité : si le filtre est vide (mots trop courts), on laisse passer le match
        if (empty($wildcardTerms)) {
            return "($searchClause * $rank)";
        }

        return "(CASE WHEN $filterClause > 0 THEN $searchClause * $rank ELSE 0 END)";
    }
}
