<?php

namespace Inside\Search\Database\Engines;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Models\Content;
use Inside\Search\Database\Models\Filter;
use Inside\Search\Database\Services\IndexerService;
use Inside\Search\Database\Services\ModelService;
use Inside\Search\Database\Services\SearcherService;
use Laravel\Scout\Builder;
use Laravel\Scout\Engines\Engine;

/**
 * Database Engine
 *
 * @category Class
 * @package  Inside\Search\Database\Engines\DatabaseEngine
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class DatabaseEngine extends Engine
{
    protected ModelService $modelService;

    protected IndexerService $indexerService;

    protected SearcherService $searcherService;

    protected ?array $orders = null;

    protected bool $isMultiModelSearch = false;

    public function __construct()
    {
        $this->modelService = app(ModelService::class);
        $this->indexerService = app(IndexerService::class);
        $this->searcherService = app(SearcherService::class);
    }

    /**
     * Update the given model in the index.
     *
     * @param  Collection  $models
     *
     * @return void
     */
    public function update($models): void
    {
        if ($models->isEmpty()) {
            return;
        }

        $this->indexerService->index($models);
    }

    /**
     * Remove the given model from the index.
     *
     * @param  Collection  $models
     *
     * @return void
     */
    public function delete($models): void
    {
        if ($models->isEmpty()) {
            return;
        }

        $this->indexerService->remove(collect($models));
    }

    /**
     * Get Filters from $builder
     *
     * @param  Builder  $builder
     * @return array
     */
    protected function filters(Builder $builder): array
    {
        return collect($builder->wheres)->map(
            function ($value, $expression) {
                $operator = null;

                if (Str::endsWith($expression, ' IN')) {
                    $expression = Str::substr($expression, 0, -3);
                    $value = explode('","', Str::substr(trim($value), 2, -2));
                    $operator = 'in';
                }
                if ($expression === 'indexable_type' && is_array($value) && count($value) > 1) {
                    $this->isMultiModelSearch = true;
                }

                $searchExpression = new Filter($expression, $value, $operator);

                return $searchExpression->get();
            }
        )->filter()->all();
    }

    /**
     * Get orders from $builder
     *
     * @param  Builder  $builder
     * @return array
     */
    protected function orders(Builder $builder): array
    {
        return collect($builder->orders)
            ->mapWithKeys(fn ($value) => [$value['column'] => $value['direction']])
            ->all();
    }

    protected function shouldNotRun(Builder $builder): bool
    {
        return strlen($builder->query) < config('scout.database.min_search_length', 3);
    }

    /**
     * Process search using $builder & $options
     *
     * @param  Builder  $builder
     * @param  array  $options
     *
     * @return mixed
     */
    protected function processSearch(Builder $builder, array $options): mixed
    {
        $result = [];

        if ($this->shouldNotRun($builder)) {
            $result['results'] = Collection::make();
            $result['count'] = 0;

            return $result;
        }

        $options['orders'] = $this->orders = $this->orders($builder);

        $options['index'] = $builder->index ?: $builder->model->searchableAs();

        if ($builder->callback) {
            return call_user_func($builder->callback, $this->searcherService, $builder->query, $options);
        }

        return $this->searcherService->search($builder->query, $options);
    }

    /**
     * Perform the given search on the engine.
     *
     * @param  Builder  $builder
     *
     * @return mixed
     */
    public function search(Builder $builder): mixed
    {
        return $this->processSearch(
            $builder,
            [
                'filters' => $this->filters($builder),
                'offset' => 0,
                'limit' => $builder->limit,
            ]
        );
    }

    /**
     * Perform the given search on the engine with pagination!
     *
     * @param  Builder  $builder
     * @param  int  $perPage
     * @param  int  $page
     *
     * @return mixed
     */
    public function paginate(Builder $builder, $perPage, $page): mixed
    {
        return $this->processSearch(
            $builder,
            [
                'filters' => $this->filters($builder),
                'offset' => ($perPage * $page) - $perPage,
                'limit' => $perPage,
            ]
        );
    }

    /**
     * Pluck and return the primary keys of the given results.
     *
     * @param mixed $results
     *
     * @return Collection
     */
    public function mapIds($results): Collection
    {
        return $results['results']->pluck('id');
    }

    /**
     * Map the given results to instances of the given model.
     *
     * @param Builder $builder
     * @param mixed $results
     * @param Model $model
     *
     * @return \Illuminate\Support\Collection
     */
    public function map(Builder $builder, $results, $model): \Illuminate\Support\Collection
    {
        if ($results['count'] === 0) {
            return collect();
        }
        $entities = collect();

        foreach ($results['results'] as $result) {
            $class = '\\'.$result['type'];
            try {
                /** @var Content $entity */
                $entity = $class::findOrFail($result['id']);

                // Adding slug
                if (isset($entity->slug)) {
                    $entity->url = count($entity->slug) > 0 ? $entity->slug[0] : '';
                }

                if (!isset($entity->type)) {
                    $entity->type = class_to_type($class);
                }

                // Add our score
                $entity->score = $result['score'];

                $entities[] = $entity;
            } catch (\Exception $e) {
                // Try to map a content that permission forbid us to get
                Log::debug("[DatabaseEngine::map] failed ".$e->getMessage());
            }
        }

        // Reorder results ( only on multiple model search, otherwise it is already sorted
        if ($this->isMultiModelSearch && is_array($this->orders)) {
            $orders = Arr::except($this->orders, ['score', 'title', 'status', 'created_at', 'updated_at']);
            if (!empty($orders)) {
                // We have a custom order and must reorder ( it takes time, should be avoid ! )
                foreach ($this->orders as $column => $direction) {
                    if (Str::lower($direction) == 'desc') {
                        $entities->sortByDesc($column);
                    } else {
                        $entities->sortBy($column);
                    }
                }
            }
        }

        return $entities;
    }

    /**
     * Get the total count from a raw result returned by the engine.
     *
     * @param  mixed  $results
     *
     * @return int
     */
    public function getTotalCount($results): int
    {
        return $results['count'];
    }

    public function flush($model): void
    {
        // TODO method to flush all models
    }
}
