<?php

namespace Inside\Search\Database\Support;

use Illuminate\Support\Facades\DB;
use Inside\Support\Str;

class PgsqlFullTextSupport extends FullTextSupport
{
    public static function listCurrentIndexes(string $table): array
    {
        $result = DB::select('SELECT indexname FROM pg_indexes WHERE tablename = :table', compact('table'));

        return collect($result)
            ->map(fn (object $index) => $index->indexname)
            ->filter(fn (string $index) => is_int(strpos($index, $table)))
            ->map(fn (string $index) => (string) str($index)->after($table.'_')->before('_index'))
            ->values()
            ->toArray();
    }

    public static function createIndex(string $table, string $name): bool
    {
        $index = "{$table}_{$name}";

        return DB::statement("CREATE INDEX {$index} ON {$table} USING gin(to_tsvector('simple', '{$name}'))");
    }

    public static function buildMatchQuery(string $column, string $search, int $rank): string
    {
        $search = str_replace(' ', ' & ', $search);

        $query = "ts_rank(to_tsvector({$column}), to_tsquery('{$search}'))";

        return "(CASE WHEN {$query} != 0 THEN {$query} * {$rank} ELSE 0 END)";
    }

    public static function buildLikeQuery(string $column, string $search, int $rank): string
    {
        $search = str_replace('%', '', $search);

        $query = "{$column} ~* '{$search}'";

        return "(CASE WHEN {$query} THEN {$rank} ELSE 0 END)";
    }
}
