<?php

namespace Inside\Search\Database\Services;

use Doctrine\DBAL\Schema\Table;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Collection;
use Inside\Content\Models\Content;
use Inside\Facades\Package;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Illuminate\Support\Facades\Schema as LaravelSchema;
use Inside\Content\Models\Field;
use Doctrine\DBAL\Schema\Index as DoctrineIndex;
use Inside\Search\Database\Contracts\FullText;
use Inside\Search\Database\Models\Index;
use Inside\Search\Facades\Searchable;
use Inside\Search\Services\Search;
use libphonenumber\PhoneNumberUtil;

/**
 * Our indexer
 *
 * @category Class
 * @package  Inside\Search\Database\Services\IndexerService
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class IndexerService
{
    public function __construct(
        protected ModelService $modelService,
        protected FullText $fullText
    ) {
    }

    /**
     *  Set the current model
     *
     * @param string $model
     * @return IndexerService
     */
    public function setModel(string $model): IndexerService
    {
        $this->modelService->setModel($model);

        return $this;
    }

    public function getFullText(): FullText
    {
        return $this->fullText;
    }

    /**
     * Create or Update index for the model
     */
    public function createOrUpdateIndex(): void
    {
        if ($this->indexAlreadyExists()) {
            return;
        }

        $this->createIndex();
    }

    /**
     * create index for the current model
     *
     */
    protected function createIndex(): void
    {
        $tableName = $this->modelService->tablePrefixedName;
        $indexes = $this->modelService->getFullTextIndexFields();

        collect($indexes)->flatten()->each(fn (string $name) => $this->fullText->createIndex($tableName, $name));
    }

    /**
     * Says if we already has index on the model
     *
     * @return bool
     */
    protected function indexAlreadyExists(): bool
    {
        $tableName = $this->modelService->tablePrefixedName;
        $neededIndexes = collect($this->modelService->getFullTextIndexFields())->keys();

        $wantedIndexes = $neededIndexes->filter(fn (string $name) => $this->fullText->hasIndex($tableName, $name));

        if ($wantedIndexes->count() !== $neededIndexes->count()) {
            Log::debug('Partial index detection : recreate');
            $this->fullText->dropIndexes($tableName, $neededIndexes->toArray());

            return false;
        }

        return true;
    }

    /**
     *
     * @param string $data
     *
     * @return Collection
     */
    protected function getTokens(string $data): Collection
    {
        $col = Search::cleanTerms($data);

        $data = $col;

        // 1. Nettoyer le HTML (Crucial pour le body longtext)
        $data = strip_tags($data);

        // 2. Nettoyer la ponctuation et les espaces insécables
        $data = preg_replace('/[[:punct:] ]/u', ' ', $data) ?? '';
        $data = preg_replace('/\s+/', ' ', $data) ?? '';

        // La chaîne nettoyée est maintenant dans $data

        // Split into words
        $words = explode(' ', $data);

        $explodes = collect();
        /** @var string $word */
        foreach ($words as $word) {
            // Only keep words that ar min 3 chars long
            if (strlen($word) >= 3) {
                // Make it lower to avoid doublon with case
                $word = Str::lower($word);

                if (!$explodes->has($word)) {
                    $explodes[$word] = 1;
                    continue;
                }

                $explodes[$word] += 1; // count occurancy
            }
        }

        return $explodes;
    }

    /**
     * Get all combination from a phrase
     *
     * @param Model  $model
     * @param string $column
     * @return Collection
     */
    protected function getCombinationPhrases(Model $model, string $column): Collection
    {
        $combinations = collect();
        $value        = $model->{$column};

        if (!is_string($value)) {
            return $combinations;
        }

        // Règle 1 : Limitation de la taille pour éviter de trop travailler sur les longs textes.
        $value = Str::lower(trim(substr(strip_tags($value), 0, 2000)));

        $value = preg_replace('/[[:punct:] ]/u', ' ', $value) ?? '';
        $value = preg_replace('/\s+/', ' ', $value) ?? '';

        $words = explode(' ', $value);
        $words = array_filter($words); // Éliminer les espaces multiples

        $stopWords = config('stopWords.words', []);
        $maxWords  = 5; // Limiter la longueur maximale des phrases générées (n-grams max 5)

        // --- NOUVELLE LOGIQUE LINÉAIRE (n-grams) ---
        $wordCount = count($words);

        // Génère des phrases de 1 mot, 2 mots, jusqu'à $maxWords
        for ($n = 1; $n <= $maxWords; $n++) { // n = longueur de la phrase (n-gram)
            for ($i = 0; $i <= $wordCount - $n; $i++) {

                // Extrait le segment de la phrase
                $phrase = implode(' ', array_slice($words, $i, $n));

                // Règle 2 : Vérification de la longueur et des stop words
                if (strlen($phrase) < 3 || in_array($phrase, $stopWords)) {
                    // On pourrait ajouter une vérification pour les stop words au milieu si nécessaire,
                    // mais pour l'instant on garde la vérification simple.
                    continue;
                }

                // Ajoute la phrase générée
                $combinations->push($phrase);
            }
        }
        // --- FIN NOUVELLE LOGIQUE ---

        // Si vous tenez à supprimer les doublons (même si l'approche linéaire les minimise)
        $combinations = $combinations->unique();

        // Vous pouvez simplifier ou supprimer la boucle "Format combinations" car elle est intégrée ci-dessus.

        return $combinations;
    }

    /**
     * Get Tokens for a given column, default to * ( means all searchable columns )
     *
     * @param Content $model
     * @param Collection|array|string $column
     *
     * @return Collection
     * @throws Exception
     */
    protected function getTokensToIndex(Content $model, $column = '*'): Collection
    {
        if (empty($column)) {
            $column = [];
        }
        $tokens = collect();
        if ($column === '*') {
            $fieldNames = $model->getSearchableColumns();
        } else {
            $fieldNames = $column;
            if (!is_array($fieldNames) && !$fieldNames instanceof Collection) {
                $fieldNames = [$fieldNames];
            }
        }

        $skipNames = false;
        $type      = class_to_type($model);
        if ($type === 'users') {
            if (isset($model->firstname)
                && isset($model->lastname)
                && !empty($model->firstname)
                && !empty($model->lastname)
            ) {
                $tokens[trim(Str::lower($model->firstname) . ' ' . Str::lower($model->lastname))] = 1;
                $tokens[trim(Str::lower($model->lastname) . ' ' . Str::lower($model->firstname))] = 1;
            }
            $options = Schema::getModelOptions('users');
            if (isset($model->email)
                && (!isset($options['search_users_email'])
                    || $options['search_users_email'] == 1)
            ) {
                $tokens[trim(Str::lower($model->email))] = 1;
            }
            $skipNames = true;
        }

        foreach ($fieldNames as $fieldName) {
            $fieldOptions = Schema::getFieldOptions($type, $fieldName);
            $value = $model->{$fieldName};
            if ($fieldOptions['type'] === 'reference') {
                $value = $model->{Str::camel($fieldName)};
            }
            if ($skipNames && in_array($fieldName, ['firstname', 'lastname', 'email']) || is_null($value)) {
                continue;
            }

            switch ($fieldOptions['type']) {
                case 'text':
                    switch ($fieldOptions['widget']) {
                        case 'phone':
                            if (is_string($value) && strlen($value) > 0) {
                                $value           = (string) preg_replace('/[^\d+]/', '', $value);
                                $phoneUtil       = PhoneNumberUtil::getInstance();
                                $shortNumberUtil = \libphonenumber\ShortNumberInfo::getInstance();
                                try {
                                    $number = $phoneUtil->parse($value, 'FR');
                                } catch (\libphonenumber\NumberParseException $e) {
                                    Log::error(
                                        '[Searchable] phone number failed to be parsed {' . $value . '} => '
                                        . $e->getMessage()
                                    );
                                    break;
                                }
                                $phoneE164              =
                                    $phoneUtil->format($number, \libphonenumber\PhoneNumberFormat::E164);
                                $phoneNational          = str_replace(
                                    ' ',
                                    '',
                                    $phoneUtil->format($number, \libphonenumber\PhoneNumberFormat::NATIONAL)
                                );
                                /** @var int $shortNumberLength */
                                $shortNumberLength      = $fieldOptions['short_number_length'] ?? 4;
                                $tokens[$phoneE164]     = 1;
                                $tokens[$phoneNational] = 1;
                                if ($shortNumberLength > 0 && strlen($phoneE164) >= $shortNumberLength) {
                                    $shortNumber          = substr($phoneE164, -$shortNumberLength);
                                    $tokens[$shortNumber] = 1;
                                }
                            }
                            continue 3; // 3 -> first switch, second switch => foreach
                        default:
                            $analysed = $this->getTokens($value);

                            if ($analysed->count() > 0) {
                                $mergedTokens = $tokens->merge($analysed);

                                $tokens = $mergedTokens->map(function ($value, $key) use ($tokens, $analysed) {
                                    return ($tokens[$key] ?? 0) + ($analysed[$key] ?? 0);
                                })->filter(function ($value) {
                                    return $value > 0;
                                });
                            }
                    }
                    break;
                case 'reference':
                    if ($value instanceof \Illuminate\Support\Collection) {
                        foreach ($value as $item) {
                            if ($item instanceof \Inside\Content\Models\Contents\Users) {
                                $tokens = $tokens->merge(
                                    [
                                        trim(Str::lower($item->firstname).' '.Str::lower($item->lastname)) => 20,
                                        trim(Str::lower($item->lastname).' '.Str::lower($item->firstname)) => 50,
                                    ]
                                );
                                continue;
                            }
                            $tokens = $tokens->union($this->getTokensToIndex($item, ['title']));
                        }
                    }
                    break;
                case 'file':
                    $content = $this->getTokens($value. " ". $this->getFileContent($value));
                    $tokens = $tokens->merge($content);
                    break;
                default:
                    $analysed = $this->getTokens($value);
                    if ($analysed->count() > 0) {
                        $mergedTokens = $tokens->merge($analysed);

                        $tokens = $mergedTokens->map(function ($value, $key) use ($tokens, $analysed) {
                            return ($tokens[$key] ?? 0) + ($analysed[$key] ?? 0);
                        })->filter(function ($value) {
                            return $value > 0;
                        });
                    }
            }
        }

        // Get sections
        if (!$model->sectionContent || ($column != '*')) {
            return $tokens;
        }
        foreach ($model->sectionContent as $content) {
            $fields = Field::whereHas(
                'model',
                function ($query) use ($content) {
                    $query->whereClass(get_class($content));
                }
            )->get();

            if ($fields->count()) {
                /** @var Field $field */

                foreach ($fields as $field) {
                    if (isset($field->options['searchable']) && $field->options['searchable']) {
                        $value = $content->{$field->name};
                        if (!is_string($value)) {
                            continue;
                        }

                        if ($field->type == 'file') {
                            try {
                                $value .= " ".$this->getFileContent($value);
                            } catch (Exception $e) {
                                Log::error(
                                    "[IndexerService::getTokensToIndex] Can't get file content [$value] {"
                                    . $e->getMessage() . "}"
                                );
                            }
                        }

                        $analysed = $this->getTokens($value);

                        if ($analysed->count() > 0) {
                            $mergedTokens = $tokens->merge($analysed);

                            $tokens = $mergedTokens->map(function ($value, $key) use ($tokens, $analysed) {
                                return ($tokens[$key] ?? 0) + ($analysed[$key] ?? 0);
                            })->filter(function ($value) {
                                return $value > 0;
                            });
                        }
                    }
                }
            }
        }

        return $tokens;
    }

    /**
     *  Prepare tokens for insertion
     *
     * @param Collection $tokens
     * @param bool $filter filter with stopwords
     *
     * @return string
     */
    protected function prepareTokens(Collection $tokens, bool $filter): string
    {
        if ($filter) {
            $stopwords = config('stopwords.words', []);
            $tokens    = $tokens->filter(
                function ($value, $key) use ($stopwords) {
                    return strlen($key) >= 3 && !in_array($key, $stopwords);
                }
            );
        }
        $items = $tokens->toArray();
        ksort($items, SORT_REGULAR);
        $tokens = collect($items);

        return $tokens->map(
            function ($times, $key) {
                return substr(str_repeat($key . "\n", $times), 0, -1);
            }
        )->filter(
            function ($value) {
                return strlen($value) >= 3;
            }
        )->implode("\n");
    }

    /**
     * Prepare Filter column with filters field from model
     *
     * @param Content $model
     *
     * @return string
     */
    protected function prepareFilters(Content $model): string
    {
        // Get filter columns
        $fields = Field::whereHas(
            'model',
            function ($query) use ($model) {
                $query->whereClass(get_class($model));
            }
        )->get();

        $filter = $fields->filter(
            function ($field) {
                $options = $field->options;

                return (isset($options['searchable_filter']) && $options['searchable_filter'])
                    || ($field->name === 'status');
            }
        )->map(
            function ($field) use ($model) {
                if ($field->type == 'reference') {
                    $references = $model->{Str::camel($field->name)};
                    if ($references !== null) {
                        $filter = [];
                        foreach ($references as $reference) {
                            $filter[] = $field->name . ':' . $reference->uuid;
                        }

                        return implode(' ', $filter);
                    }
                } else {
                    return $field->name . ':' . $this->getFilterValue($model, $field);
                }
            }
        )->implode(" ");

        if (Package::has('inside-archive')) {
            $archiveService = new \Inside\Archive\Services\ArchiveService();
            $filter .= ' archived:' . (int) $archiveService->isArchived(get_class($model), $model->uuid);
        }

        if (Package::has('inside-workflow')) {
            $service = new \Inside\Workflow\Services\ProposalService();
            $proposal = $service->getFromContent($model->uuid, get_class($model));

            if ($proposal) {
                $filter .= ' workflow_status:' . (int) $proposal->status;
            }
        }

        return trim($filter);
    }

    /**
     * Correcly format value for filter field
     *
     * @param Content $model
     * @param Field $field
     *
     * @return mixed
     */
    protected function getFilterValue(Content $model, Field $field)
    {
        switch ($field->type) {
            // TODO : check if other type needs reformating
            case 'reference':
                return $model->{Str::camel($field->name)}->implode("uuid");
            case 'checkbox':
            case 'boolean':
                return (int)$model->{$field->name};
            default:
                return $model->{$field->name};
        }
    }

    /**
     * Index entities to our index table
     *
     * @param \Illuminate\Support\Collection $entities
     */
    public function index(\Illuminate\Support\Collection $entities): void
    {
        if ($entities->isEmpty()) {
            return;
        }

        Index::withoutSyncingToSearch(
            function () use ($entities) {

                // Let's check that our index table is fullindexed ?
                $this->setModel(Index::class);
                $this->createOrUpdateIndex();

                // Get searchable classes
                $searchables = Searchable::getSearchableClasses();

                // Insert in our index table
                $entities->each(
                    function ($model) use ($searchables) {
                        $then = microtime(true);

                        // Check model
                        if ($searchables->has($model->content_type)) {
                            $tokens = $this->getTokensToIndex($model);

                            $definition = $searchables[$model->content_type];

                            $importantColumn = $model->content_type === 'users' ? 'name' : 'title';

                            $importantTokens = $this->getTokensToIndex($model, $importantColumn);

                            $startCombination            = microtime(true);

                            $importantCombinationPhrases = $this->getCombinationPhrases($model, $importantColumn);

                            $nowCombination = microtime(true);

                            Log::debug(
                                "[" . get_class($model) . "] {" . $model->uuid . "} " . "Indexed in [" . sprintf(
                                    "Combination done in:  %f",
                                    $nowCombination - $startCombination
                                ) . "]"
                            );

                            // Prepare filter column !
                            $filters = $this->prepareFilters($model);

                            // Only index if we have some contents to index
                            if ($tokens->count() > 0 || $importantTokens->count() > 0) {
                                $model->index()->delete();

                                $langcode = null;
                                if ($model->isTranslatable && class_to_type($model) !== 'users') {
                                    $langcode = $model->langcode;
                                }

                                $model->index()->create(
                                    [
                                        'langcode'          => $langcode,
                                        'content'           => $this->prepareTokens($tokens, true),
                                        'filter'            => $filters,
                                        'important_content' => $importantCombinationPhrases->implode("\n") . "\n"
                                            . $this->prepareTokens($importantTokens, false),
                                        // We don't want to filter stopwords on the important column
                                        'title'             => ($model->content_type !== 'users')
                                            ? $model->title
                                            : (Str::upper($model->lastname) . ' ' . Str::ucfirst(
                                                Str::lower($model->firstname)
                                            )),
                                        'published'         => $model->status,
                                        'created_at'        => $model->created_at,
                                        'updated_at'        => $model->updated_at,
                                        'published_at'      => $model->published_at,
                                    ]
                                );
                            }
                        }
                        $now = microtime(true);

                        Log::debug(
                            "[" . get_class($model) . "] {" . $model->uuid . "} " . "Indexed in [" . sprintf(
                                "Elapsed:  %f",
                                $now - $then
                            ) . "]"
                        );
                    }
                );
            }
        );
    }

    /**
     * Remove entities from our index
     *
     * @param Collection $entities
     */
    public function remove(Collection $entities): void
    {
        if ($entities->isEmpty()) {
            return;
        }

        // $entities is supposed to be a collection of one same Model type
        $model    = $entities->first(); // So we can use first model as a template
        $relation = $model->index();

        try {
            Index::where(
                $relation->getMorphType(),
                $relation->getMorphClass()
            )->whereIn(
                $relation->getForeignKeyName(),
                $entities->pluck('uuid')->all()
            )->delete();
        } catch (Exception $e) {
        }
    }

    /**
     * Using Tika to get File content
     *
     * @param string $path
     *
     * @return string
     */
    protected function getFileContent(string $path): string
    {
        if (empty($path) || !config('tika.enabled')) {
            return '';
        }

        try {
            setlocale(LC_ALL, "");
            $client = \Vaites\ApacheTika\Client::make(config('tika.host'));

            $realpath = Storage::disk('local')->path($path);
            // Convert path using local machine separators & get content
            $text = $client->getText(str_replace('/', DIRECTORY_SEPARATOR, $realpath));
        } catch (Exception $e) {
            Log::warning('Failed to load file content ['.$path.'] with Tika ['.$e->getMessage().']');

            return '';
        }

        return $text ?? '';
    }
}
