<?php

namespace Inside\Search\Database\Models;

use Inside\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

/**
 * FilterItem Model
 *
 * @category Class
 * @package  Inside\Search\Database\Models\FilterItem
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class FilterItem
{
    /**
     * @var string
     */
    private $expression;

    /**
     * @var string|null
     */
    private $operator;

    /**
     * @var mixed
     */
    private $value;

    /**
     * FilterItem constructor.
     *
     * @param string $expression
     * @param string|null $operator
     * @param mixed $value
     */
    public function __construct(string $expression, string $operator = null, $value = null)
    {
        $this->expression = $expression;
        $this->operator   = $operator;
        $this->value      = $value;
    }

    /**
     *
     * @param Builder $query
     */
    public function apply(string $table, Builder $query): void
    {
        if (is_array($this->value)) {
            if ($this->expression === 'or') {
                $query->where(function ($query) {
                    foreach ($this->value as $orFilter) {
                        $query->orWhere(function ($q) use ($orFilter) {
                            foreach ($orFilter as $key => $value) {
                                if (str_contains($key, ':')) {
                                    [$field, $operator] = explode(':', $key);
                                    if (!in_array($field, ['created_at', 'updated_at', 'publishable_at'])) {
                                        continue;
                                    }

                                    $operator = match ($operator) {
                                        'lt' => '<',
                                        'lte' => '<=',
                                        'gte' => '>=',
                                        default => '>'
                                    };
                                    $q->where($field, $operator, Carbon::parse($value));
                                } else {
                                    if ($key == 'status') {
                                        $q->whereLike('filter', "%$key:$value%");
                                    } else {
                                        $q->where($key, $value);
                                    }
                                }
                            }
                        });
                    }
                });
                return;
            }

            if (in_array(null, $this->value)) {
                $query->where(
                    function ($query) use ($table) {
                        $query->whereNull($table . '.'  . $this->expression)->orWhereIn(
                            $table . '.'  . $this->expression,
                            array_filter($this->value)
                        );
                    }
                );
                return;
            }

            $query->whereIn($table . '.'  . $this->expression, $this->value);
            return;
        }
        if (null === $this->operator) {
            $query->whereRaw($table . '.'  . $this->expression, $this->value ?: []);

            return;
        }

        if (strpos($this->value, '&')) {
            // We need a AND in this case with same field name
            $values = explode('&', $this->value);
            foreach ($values as $value) {
                if (strpos($value, '|')) {
                    $orValues = explode('|', $value);
                    $query->where(
                        function ($query) use ($orValues, $table) {
                            foreach ($orValues as $orValue) {
                                $query->orWhere($table . '.'  . $this->expression, $this->operator, $orValue);
                            }
                        }
                    );
                    continue;
                }
                $query->where($table . '.'  . $this->expression, $this->operator, $value);
            }

            return;
        }
        if (strpos($this->value, '|')) {
            $orValues = explode('|', $this->value);
            $query->where(
                function ($query) use ($orValues, $table) {
                    foreach ($orValues as $orValue) {
                        $query->orWhere($table . '.'  . $this->expression, $this->operator, $orValue);
                    }
                }
            );

            return;
        }

        $columns = explode(',', str_replace(' ', '', $this->expression));

        $query->where(
            function (Builder $query) use ($columns, $table) {
                foreach ($columns as $column) {
                    if ($this->operator !== 'in') {
                        $query->orWhere($table . '.'  . $column, $this->operator, $this->value);
                        continue;
                    }

                    $values = $this->value;
                    if (!is_array($values)) {
                        $values = preg_split(
                            "/\s*\",\"\s*/",
                            preg_replace(["/\s*\(\s*\"?/", "/\s*\"?\)\s*/"], "", $values)
                        );
                    }

                    $query->orWhereIn($table . '.'  . $column, $values);
                }
            }
        );
    }

    public function getExpression(): string
    {
        return $this->expression;
    }

    /**
     * @return null
     */
    public function getValue()
    {
        return $this->value ?? $this->operator;
    }

    /**
     * @return mixed
     */
    public function getOperator()
    {
        return $this->operator;
    }
}
