<?php

namespace Inside\Search\Database\Models;

use Illuminate\Support\Str;

/**
 * Filter Model
 *
 * @category Class
 * @package  Inside\Search\Database\Models\Filter
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class Filter
{
    /** @var string */
    private $grammar;

    /** @var mixed */
    private $value;

    /** @var FilterItem */
    private $result;

    /** @var string|null */
    private $operator;

    /**
     * Filter constructor.
     *
     * @param string $grammar
     * @param mixed $value
     * @param string|null $operator
     * @throws \Exception
     */
    public function __construct(string $grammar, $value, string $operator = null)
    {
        $this->grammar = $grammar;
        $this->value   = $value;
        $this->operator = $operator;

        $this->build();
    }

    /**
     * @return mixed
     */
    public function get()
    {
        return $this->result;
    }

    /**
     * @return FilterItem
     * @throws \Exception
     */
    private function build()
    {
        $grammar      = trim($this->grammar);
        $lowerGrammar = strtolower($grammar);
        $chars        = str_replace(' ', '', $lowerGrammar);

        if (null === $this->value || is_array($this->value)) {
            return $this->result = new FilterItem($grammar, $this->operator, $this->value);
        }

        $normalizedGrammar = str_replace(['  ', ', '], [' ', ','], $lowerGrammar);

        $checkEndedChars   = [
            ['!~~*'],
            ['<=>', '!~*', '~~*'],
            ['<=', '>=', '<>', '!=', '<<', '>>', '~*', '!~', 'in'],
            ['=', '<', '>', '&', '|', '^', '~'],
        ];

        foreach ($checkEndedChars as $checkEndedChar) {
            $length     = -(strlen($checkEndedChar[0]));
            $endedChars = substr($chars, $length);

            if (in_array($endedChars, $checkEndedChar, true)) {
                $expression = rtrim(substr($grammar, 0, $length));

                return $this->result = new FilterItem($expression, $endedChars, $this->value);
            }
        }

        $expression = $grammar;
        $operator   = '=';

        $grammarCheck = [
            ' not like',
            ' like',
            ' not ilike',
            ' ilike',
            ' not rlike',
            ' rlike',
            ' not like binary',
            ' like binary',
            ' not between',
            ' between',
            ' not regexp',
            ' regexp',
            ' not similar to',
            ' similar to',
        ];

        foreach ($grammarCheck as $check) {
            $length = -(strlen($check));
            if (substr($normalizedGrammar, $length) === $check) {
                $expression = rtrim(substr($grammar, 0, $length));
                $operator   = trim($check);
                break;
            }
        }

        return $this->result = new FilterItem($expression, $operator, $this->value);
    }
}
