<?php

namespace Inside\RSS\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;
use Inside\Permission\Facades\Permission;
use Inside\RSS\Helpers\RssFeeder;
use Laravel\Lumen\Http\ResponseFactory;
use Illuminate\Http\Response;

/**
 * RSS controller class.
 *
 * @category Class
 * @package  Inside\RSS\Http\Controllers
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class RssController
{
    /**
     * View datas from RSS flux
     *
     * @param Request $request
     * @return array
     */
    public function view(Request $request)
    {
        $flux = null;
        $fields = [];
        $filters = [];

        if ($request->has('flux')) {
            $flux = json_decode($request->get('flux'), true);
            if (empty($flux)) {
                $flux = $request->get('flux');
            }
        }

        if ($request->has('fields')) {
            $fields = json_decode($request->get('fields'), true);
        }

        if ($request->has('filters')) {
            $filters = json_decode($request->get('filters'), true);
        }

        $appKey = env('APP_KEY');
        if (Cache::has('inside.' . $appKey . '.rss.' . $flux)) {
            return Cache::get('inside.' . $appKey . '.rss.' . $flux);
        }

        try {
            $feeder = new RssFeeder($flux, $fields, $filters);

            $data = $feeder->feed();
        } catch (\Exception $e) {
            $data = [];
        }

        // Remember it 2 hours
        Cache::put('inside.' . $appKey . '.rss.' . $flux, $data, intval(config('feed.cache_max_age')));

        return $data;
    }

    /**
     * @param string $contentType
     *
     * @return Response
     */
    public function feed(string $contentType): Response
    {
        if (!config('feed.is_active')) {
            return response('error : Rss feed is disabled.', 404);
        }

        $fields = Arr::except(config('feed.content_type.'.$contentType), ['custom_query']);

        if (!$fields) {
            return response('error : '. $contentType .' content type is not set.', 404);
        }

        $limit = config('feed.limit', 200);

        Permission::disableAllowedScope();
        $results = [];
        $query = type_to_class($contentType)::where('status', 1)->orderBy('updated_at', 'desc')->limit($limit);

        $filter = config('feed.content_type.'.$contentType.'.custom_query');
        if ($filter && is_callable($filter)) {
            $query = $filter($query);
        }

        $contents = $query->get();

        foreach ($contents as $key => $content) {
            $results[$key] = $content->only($fields);
            $results[$key] = array_merge($results[$key], [
                'image' => $this->imageFeed($content),
                'slug' => env("APP_URL")."/".$content->slug[0]
            ]);
        }

        Permission::enableAllowedScope();
        return response(view("rss::rss.feed", compact('results')), 200, ['Content-Type' => 'text/xml']);
    }

    /**
     * @param mixed $content
     * @return string|null
     */
    protected function imageFeed($content): ?string
    {
        if (!isset($content->image)) {
            return null;
        }

        $imageName = explode('/', $content->image);
        $imageName = Arr::last($imageName);

        if (!Storage::disk('public')->exists($imageName)) {
            if (!Storage::disk('local')->exists($content->image)) {
                return null;
            }
            Storage::copy($content->image, 'public/'.$imageName);
        }

        return env('APP_URL').'/images/'.$imageName;
    }
}
