<?php

namespace Inside\RSS\Helpers;

use Symfony\Component\Config\Util\Exception\InvalidXmlException;
use Symfony\Component\Config\Util\Exception\XmlParsingException;
use IntlDateFormatter;

/**
 * Rss Feeder class.
 * Refer to http://www.intertwingly.net/wiki/pie/Rss20AndAtom10Compared for different syndications
 *
 * @category Class
 * @package  Inside\RSS\Helpers
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class RssFeeder
{
    /**
     * Syndication type Atom 1.0
     */
    private const ATOM10 = 1;

    /**
     * Syndication type RSS 2.0
     */
    private const RSS20  = 2;

    /**
     * The sindycation type
     *
     * @var int
     */
    protected $syndication = self::RSS20;

    /**
     * The feed URL
     *
     * @var array|string|null
     */
    protected $feed = '';

    /**
     * Fields returned
     *
     * @var array
     */
    protected $fields = ['title', 'link'];

    /**
     * Filters
     *
     * @var array
     */
    protected $filters = [];

    /** @var bool|array */
    private $trustedDomains = false;

    /**
     * Construct new RSS Feeder instance
     *
     * @param string|array|null $feed
     * @param array $fields
     * @param array $filters
     */
    public function __construct($feed = null, array $fields = [], array $filters = [])
    {
        $this->feed = $feed;

        if (!empty($fields)) {
            $this->fields = $fields;
        }

        if (!empty($filters)) {
            $this->filters = $filters;
        }

        if (!empty(env('RSS_TRUSTED_DOMAINS'))) {
            $this->trustedDomains = explode(',', env('RSS_TRUSTED_DOMAINS', ''));
        }
    }

    /**
     * Return array of datas using $fields as filter of xml format file
     *
     * @param mixed $node
     * @param array $datas
     * @param array $fields
     * @param string|null $sourceLink
     * @return void
     */
    private function parser($node, array &$datas = [], array $fields = [], $sourceLink = null)
    {
        $counter = 0;
        $limit = $this->filters['limit'] ?? null;
        $source = $this->filters['source'] ?? null;

        foreach ($node as $item) {
            $data = [];

            if ($source) {
                $data['source_link'] = $sourceLink;
            }

            foreach ($fields as $key => $field) {
                $this->parseField($item, $field, $key, $data);
            }

            if ($limit && ++$counter > $limit) {
                break;
            }

            $datas[] = $data;
        }
    }

    /**
     * Parse field
     *
     * @param mixed $item
     * @param mixed $field
     * @param string $key
     * @param array $data
     * @return void
     */
    protected function parseField($item, $field, string $key, array &$data = [])
    {
        if (is_string($field) && strpos($field, ':')) {
            $fieldArray = explode(':', $field);
            $namespace = $fieldArray[0];
            $attributeName = $fieldArray[1];
            $data[$namespace] = $item->children($namespace, true)->$attributeName->__toString();
            return;
        }

        if (is_array($field)) {
            // Get attribute value of given element
            if (substr(reset($field), 0, 1) === '@') {
                $fieldKey = key($field);
                $fieldValue = substr(reset($field), 1);
                $data[$fieldKey] = $item->$fieldKey->attributes()->$fieldValue->__toString();
                return;
            }

            if (isset($item->{$key})) {
                self::parser($item->{$key}, $data[$field], $field);
                return;
            }
        }

        if (!isset($item->{$field})) {
            return;
        }

        if ($field === 'link' && $this->syndication === self::ATOM10) {
            $data[$field] = $item->{$field}->attributes()->href->__toString();
            return;
        }

        if ($field === 'pubDate' || $field == 'updated') {
            $dateTime = strtotime($item->{$field}->__toString());

            if (!$dateTime) {
                $dateFormatter = new IntlDateFormatter(
                    "fr-FR",
                    IntlDateFormatter::FULL,
                    IntlDateFormatter::FULL,
                    'Etc/UTC',
                    IntlDateFormatter::GREGORIAN
                );
                $dateString = $item->{$field}->__toString();
                $dateFilter = config('feed')['date_filter'];
                if (isset($dateFilter) && is_callable($dateFilter)) {
                    $dateString = $dateFilter($dateString);
                }

                $dateTime = $dateFormatter->parse($dateString);
            }

            $data[$field] = $dateTime;
            return;
        }

        $data[$field] = $item->{$field}->__toString();
    }

    /**
     * @param string|null $feed
     * @param mixed $node
     * @param array $datas
     * @return void
     */
    private function extractDataFromFeed(?string $feed, $node, &$datas)
    {
        if (empty($feed)) {
            throw new XmlParsingException('Feed url not provided !');
        }

        if ($this->trustedDomains && !in_array(parse_url($feed)['host'], $this->trustedDomains)) { // @phpstan-ignore-line
            throw new XmlParsingException('Url not authorized : '. $feed);
        }

        try {
            $sourceLink = null;
            $xmlStringResponse = $this->fetchXmlFeed($feed);
            $xml = simplexml_load_string($xmlStringResponse);

            if ($xml === false || (!isset($xml->entry) && !isset($xml->channel))) {
                throw new InvalidXmlException('RSS url invalid : ' . $feed, 400);
            }

            if (isset($xml->channel)) {
                $node = $xml->channel->item;
                $sourceLink = (string) $xml->channel->link;
            }

            if (isset($xml->entry)) {
                $node = $xml->entry;
                $this->syndication = self::ATOM10;
            }
        } catch (\Exception $e) {
            throw new XmlParsingException('RSS url or XML format invalid : ' . $feed, 400);
        }

        self::parser($node, $datas, $this->fields, $sourceLink);
    }

    /**
     * Return asked data of a RSS feed
     *
     * @return array
     * @throws InvalidXmlException
     * @throws XmlParsingException
     */
    public function feed(): array
    {
        $node  = null;
        $datas = [];

        if (is_array($this->feed)) {
            foreach ($this->feed as $feed) {
                $this->extractDataFromFeed($feed, $node, $datas);
            }
        } else {
            $this->extractDataFromFeed($this->feed, $node, $datas);
        }

        return $datas;
    }

    private function fetchXmlFeed(string $url): string
    {
        $ch = curl_init($url);

        if ($ch === false) {
            throw new XmlParsingException('CURL initialization failed !');
        }

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            CURLOPT_HTTPHEADER => [
                'Accept: application/xml,text/xml;q=0.9,*/*;q=0.8',
            ],
        ]);

        $response = curl_exec($ch);
        curl_close($ch);

        if ($response === false) {
            throw new XmlParsingException('CURL error : ' . curl_error($ch));
        }

        return is_string($response) ? $response : '';
    }
}
