<?php

namespace Inside\Reservation\Services;

use DateTimeZone;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Validator as IlluminateValidator;
use Inside\Content\Facades\Schema;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Reservation\Contracts\Reservation as ReservationContract;
use Inside\Reservation\Repositories\ReservationRepository;
use InvalidArgumentException;
use Recurr\Exception\InvalidRRule;
use Recurr\Exception\InvalidWeekday;
use Recurr\Recurrence;
use Recurr\Rule;
use Recurr\Transformer\ArrayTransformer;
use Throwable;

class ReservationService implements ReservationContract
{
    public const MICROSECONDS_PER_SECOND = 1000000;

    protected ?array $lastConflict = null;

    public function __construct(
        protected ContentTransformer $transformer,
        protected ReservationRepository $repository
    ) {
    }

    /**
     * @throws InvalidWeekday
     */
    public function getReservationList(
        array $types,
        int $startDate,
        int $endDate,
        array $filters = [],
        array $fields = []
    ): array {
        if (empty($fields)) {
            $fields = config('reservation.index.fields', []);
        }
        $startDate = get_date($startDate);
        $endDate = get_date($endDate);
        if (!$startDate || !$endDate) {
            throw new InvalidArgumentException('Invalid start date and end date');
        }
        $reservations = new Collection();

        foreach ($types as $type) {
            if (
                ! Schema::hasContentType($type)
                || ! Schema::hasContentType("{$type}_categories")
                || ! Schema::hasContentType("{$type}_reservations")
            ) {
                Log::error("[ReservationService::getReservationList] Type [$type] does not look like a reservation type");
                break;
            }

            if (! array_key_exists($type, $fields)) {
                $fields[$type] = [
                    'uuid',
                    'title',
                    "{$type}_category" => [
                        'uuid',
                        'title',
                        'color',
                    ],
                ];
            }

            try {
                $query = $this->repository->getReservationListQuery($type, $startDate, $endDate, $filters);
            } catch (Throwable $e) {
                continue;
            }

            $reservations = $reservations->merge($query->get()->reject(fn ($reservation) => $reservation->{$type}->isEmpty()));
        }

        foreach ($reservations as $key => $reservation) {
            if (is_null($reservation->frequency)) {
                continue;
            }

            try {
                $reservationStartDate = get_date($reservation->start_date);
                $reservationEndDate = get_date($reservation->end_date);
                if (! $reservationStartDate || ! $reservationEndDate) {
                    throw new InvalidArgumentException("Invalid start date and end date");
                }
                if ($reservation->cooldown > 0) {
                    $reservationEndDate->addMinutes($reservation->cooldown);
                }

                $exceptions = ! empty($reservation->exceptions) ? explode(',', $reservation->exceptions) : [];
                foreach ($exceptions as &$exception) {
                    $exception = get_date((int) $exception);
                }
                if (
                    $this->checkAvailabilityOnRecurse(
                        new Rule($reservation->frequency, $reservationStartDate, $reservationEndDate),
                        $startDate,
                        $endDate,
                        $exceptions
                    )
                ) {
                    $reservations->forget($key);
                }
            } catch (InvalidRRule $e) {
            }
        }

        return $this->transformer->transformCollection($reservations, $fields);
    }

    public function checkAvailabilityOnRecurse(
        Rule $rule,
        Carbon $startDate,
        Carbon $endDate,
        array $exceptions = []
    ): bool {
        // Always localize rule to Europe/Paris timezone for recurrence logic
        /** @var Carbon $startLocal */
        $startLocal = get_date_in_user_timezone($startDate);
        /** @var Carbon $endLocal */
        $endLocal = get_date_in_user_timezone($endDate);

        $timezone = $startLocal->getTimezone();

        // Replace rule's start/end with localized versions
        $rule->setStartDate($rule->getStartDate()->setTimezone($timezone))
            ->setEndDate($rule->getEndDate()?->setTimezone($timezone))
            ->setTimezone($timezone->getName());

        // Apply exception dates by filtering later instead of modifying the Rule
        $exceptionTimestamps = array_map(
            fn (Carbon $date) => $date->getTimestamp(),
            array_filter($exceptions, fn ($exDate) => $exDate instanceof Carbon)
        );

        $transformer = new ArrayTransformer();
        $recurrences = $transformer->transform($rule);

        foreach ($recurrences as $occurrence) {
            $occStartLocal = Carbon::instance($occurrence->getStart());
            $occEndLocal = Carbon::instance($occurrence->getEnd());

            // Skip occurrence if it matches an exception
            if (in_array($occStartLocal->getTimestamp(), $exceptionTimestamps, true)) {
                continue;
            }

            if (
                $startLocal->between($occStartLocal, $occEndLocal->copy()->subSecond()) ||
                $endLocal->between($occStartLocal->copy()->addSecond(), $occEndLocal) ||
                ($startLocal->lte($occStartLocal) && $endLocal->gte($occEndLocal))
            ) {
                $this->lastConflict = array_values(
                    array_merge(
                        [$occurrence],
                        [$startLocal->getOffset() / 3600]
                    )
                );

                return false;
            }
        }

        return true;
    }


    public function shiftTimezone(Carbon $carbonDate, DateTimeZone | string $value): Carbon
    {
        $offset = $carbonDate->offset;
        $date = $carbonDate->setTimezone($value);

        return $this->addRealMicroseconds($date, ($offset - $date->offset) * static::MICROSECONDS_PER_SECOND);
    }

    public function addRealMicroseconds(Carbon $carbonDate, int $value): Carbon
    {
        $diff = (int) $carbonDate->format('u') + $value;
        $time = $carbonDate->getTimestamp();
        $seconds = (int) floor($diff / static::MICROSECONDS_PER_SECOND);
        $time += $seconds;
        $diff -= $seconds * static::MICROSECONDS_PER_SECOND;
        $microTime = str_pad("$diff", 6, '0', STR_PAD_LEFT);
        $tz = $carbonDate->tz;

        return $carbonDate->tz('UTC')->modify("@$time.$microTime")->tz($tz);
    }

    /**
     * @throws InvalidWeekday
     */
    public function loadAvailableRule(): void
    {
        Validator::extend(
            'available',
            function ($attribute, $value, $parameters, $validator) {
                $uuid = null;
                if (count($parameters) >= 1) {
                    $uuid = $parameters[0];
                }
                if (! Schema::hasContentType($attribute.'_reservations')) {
                    Log::error('Trying to use available rule on a wrong type ['.$attribute.']');

                    return false;
                }

                $data = $validator->getData();
                if ($uuid !== null && ! isset($data['start_date']) && ! isset($data['end_date'])) {
                    return true;
                }

                $startDate = get_date_in_user_timezone($data['start_date']);
                $endDate = get_date_in_user_timezone($data['end_date']);

                if (! $startDate || ! $endDate) {
                    throw new InvalidArgumentException("Invalid start date and end date");
                }

                if (isset($data['cooldown'])) {
                    $endDate->addMinutes($data['cooldown']);
                }

                if (isset($data['frequency']) && ! empty($data['frequency'])) {
                    try {
                        $rule = new Rule($data['frequency'], $startDate, $endDate);
                        $transformer = new ArrayTransformer();
                        $exceptions = (isset($data['exceptions']) && ! empty($data['exceptions'])) ? explode(
                            ',',
                            $data['exceptions']
                        ) : [];
                        foreach ($exceptions as &$exception) {
                            $exception = get_date((int) $exception);
                        }

                        foreach ($transformer->transform($rule) as $date) {
                            $start = get_date($date->getStart());
                            $end = get_date($date->getEnd());
                            if (! $start || ! $end) {
                                throw new InvalidArgumentException("Invalid start date or end date !");
                            }

                            if (! $this->checkAvailability($attribute, $uuid, $value, $start, $end, $validator)) {
                                Log::debug('checkAvailability failed for ['.$start->toString().' - '.$end->toString().')');

                                return false;
                            }
                        }
                    } catch (InvalidRRule $e) {
                    }

                    return true;
                } else {
                    return $this->checkAvailability(
                        $attribute,
                        $uuid,
                        $value,
                        $startDate,
                        $endDate,
                        $validator
                    );
                }
            }
        );

        Validator::replacer(
            'available',
            function ($message, $attribute, $rule, $parameters, $validator) {
                $data = $validator->getData();
                if (! isset($data[$attribute])) {
                    throw new InvalidArgumentException('missing field ['.$attribute.']');
                }

                $entity = call_user_func(type_to_class($attribute).'::findOrFail', $data[$attribute]);
                if ($entity === null) {
                    throw new InvalidArgumentException('entity ['.$data[$attribute].'] does not exist');
                }

                App::refreshLocales();
                $title = $data['found_title'] ?? '';
                $cooldown = $data['found_cooldown'] ?? 0;

                $defaultGmt = (get_date_in_user_timezone($data['found_start_date'])?->offset / 3600);

                return [
                    'title' => $entity->title,
                    'start_date' =>  get_date($data['start_date'])?->timestamp,
                    'end_date' =>  get_date($data['end_date'])?->timestamp,
                    'conflict_title' => $title,
                    'conflict_start_date' => $data['found_start_date']->timestamp,
                    'conflict_end_date' => $data['found_end_date']->subMinutes($cooldown)->timestamp,
                    'cooldown' => $cooldown,
                    // Return the timezone offset (in hours) of the initial date of the occurrence (ex: GMT+2)
                    'gmt' => $data['gmt'] ?? $defaultGmt,
                ];
            }
        );
    }

    /**
     * @throws InvalidWeekday
     * @throws Exception
     */
    public function checkAvailability(
        string $attribute,
        ?string $uuid,
        mixed $value,
        Carbon $startDate,
        Carbon $endDate,
        IlluminateValidator $validator
    ): bool {
        $reservations = call_user_func(type_to_class($attribute.'_reservations').'::with', Str::camel($attribute));
        $reservations = $reservations->whereHas(
            Str::camel($attribute),
            function ($query) use ($value, $attribute) {
                $query->where(type_to_table($attribute).'.uuid', $value);
            }
        )->when(
            $uuid !== null,
            function ($query) use ($attribute, $uuid) {
                return $query->where(type_to_table($attribute.'_reservations').'.uuid', '!=', $uuid);
            }
        )->whereNotNull('frequency')->get();

        foreach ($reservations as $reservation) {
            $reservationStartDate = get_date($reservation->start_date);
            $reservationEndDate = get_date($reservation->end_date);
            if (!$reservationEndDate || !$reservationStartDate) {
                throw new Exception('There is a problem with the reservation starting date and end date !');
            }

            if ($reservation->cooldown > 0) {
                $reservationEndDate->addMinutes($reservation->cooldown);
            }

            $exceptions = ! empty($reservation->exceptions) ? explode(',', $reservation->exceptions) : [];
            foreach ($exceptions as &$exception) {
                $exception = get_date((int) $exception);
            }

            try {
                if (
                    ! $this->checkAvailabilityOnRecurse(
                        new Rule($reservation->frequency, $reservationStartDate, $reservationEndDate),
                        $startDate,
                        $endDate,
                        $exceptions
                    )
                ) {
                    $found = $this->getLastConflictedReservation();
                    Log::debug(
                        'Reservation ['.$reservationStartDate->toString().' - '.$reservationEndDate->toString()
                        .']{'.$reservation->frequency.'} => ('.$startDate->toString().' - '
                        .$endDate->toString().')'
                    );

                    if (! is_array($found)) {
                        throw new Exception('Error $found is not an array');
                    }

                    $recurrence = Arr::first($found);
                    $data = array_merge(
                        $validator->getData(),
                        [
                            'found_title' => $reservation->title,
                            'found_cooldown' => $reservation->cooldown,
                            'found_start_date' => get_date($recurrence->getStart()),
                            'found_end_date' => get_date($recurrence->getEnd()),
                            // Return the timezone offset (in hours) of the initial date of the occurrence (ex: GMT+2)
                            'gmt' => $found[1],
                        ]
                    );
                    $validator->setData($data);

                    return false;
                }
            } catch (InvalidRRule $e) {
                Log::error('InvalidRRule ['.$e->getMessage().']');
            }
        }

        $reservations = call_user_func(type_to_class($attribute.'_reservations').'::with', Str::camel($attribute));
        $query = $reservations->whereHas(
            Str::camel($attribute),
            function ($query) use ($value, $attribute) {
                $query->where(type_to_table($attribute).'.uuid', $value);
            }
        )->whereNull('frequency')->where(
            function ($query) use ($startDate, $endDate) {
                $query->where(
                    function ($query) use ($startDate) {
                        $query->where('start_date', '<', get_date($startDate)?->toDateTimeString())->where(
                            DB::raw('DATE_ADD(end_date, INTERVAL cooldown MINUTE)'),
                            '>',
                            get_date($startDate)?->toDateTimeString()
                        );
                    }
                )->orWhere(
                    function ($query) use ($endDate) {
                        $query->where('start_date', '<', get_date($endDate)?->toDateTimeString())->where(
                            DB::raw('DATE_ADD(end_date, INTERVAL cooldown MINUTE)'),
                            '>',
                            get_date($endDate)?->toDateTimeString()
                        );
                    }
                )->orWhere(
                    function ($query) use ($startDate, $endDate) {
                        $query->where('start_date', '>=', get_date($startDate)?->toDateTimeString())->where(
                            DB::raw('DATE_ADD(end_date, INTERVAL cooldown MINUTE)'),
                            '<=',
                            get_date($endDate)?->toDateTimeString()
                        );
                    }
                );
            }
        )->when(
            $uuid !== null,
            function ($query) use ($uuid, $attribute) {
                $query->where(type_to_table($attribute.'_reservations').'.uuid', '!=', $uuid);
            }
        )->where(type_to_table($attribute.'_reservations').'.status', true);

        if ($query->exists()) {
            $found = $query->first();
            $data = array_merge(
                $validator->getData(),
                [
                    'found_title' => $found->title,
                    'found_cooldown' => $found->cooldown,
                    'found_start_date' => get_date($found->start_date),
                    'found_end_date' => get_date($found->end_date),
                ]
            );

            $validator->setData($data);

            return false;
        }

        return true;
    }

    public function getLastConflictedReservation(): ?array
    {
        return $this->lastConflict;
    }
}
