<?php

namespace Inside\Reservation\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\AnonymousNotifiable;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Ideas;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Channels\DatabaseChannel;
use Inside\Notify\Messages\MailMessage;
use Inside\Notify\Models\NotificationType;
use ReflectionClass;

class SimpleNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public string $notificationName;

    protected bool $withIcs;

    public function __construct(
        protected Content $reservation,
        protected User $user,
        protected ?string $customSlug = null
    ) {
        $reflection = new ReflectionClass($this);
        $this->notificationName = Str::snake($reflection->getShortName());
        $this->withIcs = false;
        Log::debug('[SimpleNotification] {' . $this->notificationName . '}');
    }

    public function withIcs(): SimpleNotification
    {
        $this->withIcs = true;

        return $this;
    }

    public function via(AnonymousNotifiable|User $notifiable): array
    {
        if ($notifiable instanceof AnonymousNotifiable || !$notifiable->exists) {
            return ['mail'];
        }

        return ['mail', DatabaseChannel::class];
    }

    public function toMail(AnonymousNotifiable|User $notifiable): MailMessage
    {
        $data       = [];
        $data['to'] = [
            'uuid'      => '',
            'lastname'  => '',
            'firstname' => '',
            'fullname'  => '',
            'image'     => '',
        ];
        if (!$notifiable instanceof AnonymousNotifiable && $notifiable->exists) {
            $recipient  = Users::find($notifiable->uuid);
            $data['to'] = [
                'uuid'      => $recipient->uuid,
                'lastname'  => $recipient->lastname,
                'firstname' => $recipient->firstname,
                'fullname'  => $recipient->firstname . ' ' . $recipient->lastname,
                'image'     => $recipient->image,
            ];
        }
        $author = $this->user->information;
        if ($author) {
            $data['author'] = [
                'uuid' => $author->uuid,
                'lastname' => $author->lastname,
                'firstname' => $author->firstname,
                'fullname' => $author->firstname . ' ' . $author->lastname,
                'image' => $author->image,
            ];
        }
        $locale = $notifiable->langcode ?? config('app.locale');
        $this->setLocale($locale);
        $data['model'] = $this->reservation->toArray();
        $data['organizer'] = ($this->reservation->organizer && $this->reservation->organizer->first()) ? $this->reservation->organizer->first()->toArray() : [];
        $data['organizer']['fullname'] = ($data['organizer']['firstname'] ?? '') . ' ' . ($data['organizer']['lastname'] ?? '');
        $startDate = get_date($this->reservation->start_date);
        $endDate = get_date($this->reservation->end_date);
        if (!$startDate || !$endDate) {
            throw new \Exception('There is a problem with the given dates !');
        }
        $data['timelapse'] = $this->getTimeLapseForHumans($startDate, $endDate);

        $data['participants'] = '';
        if ($this->reservation->internalUsers) {
            foreach ($this->reservation->internalUsers as $p) {
                $data['participants'] .= $p->firstname . ' ' . $p->lastname . ' - ' . $p->email . "\n";
            }
        }
        if ($this->reservation->external_users !== null) {
            foreach (explode(',', $this->reservation->external_users) as $p) {
                $email = filter_var(trim($p), FILTER_VALIDATE_EMAIL);
                if ($email) {
                    $data['participants'] .= $email . "\n";
                }
            }
        }
        if (empty($data['participants'])) {
            $data['participants'] = Lang::get('reservation.no-participant');
        }

        $data['entity'] = $this->getEntity()->toArray();
        $data = array_map(
            'trim',
            array_filter(
                Arr::dot($data),
                function ($value) {
                    return is_string($value);
                }
            )
        );

        $subject = $this->getLanguageContent(
            'subject',
            $data,
            $locale
        );
        $content = $this->getLanguageContent(
            'content',
            $data,
            $locale
        );
        $buttonText = $this->getLanguageContent(
            'button',
            $data,
            $locale
        );
        $contents = explode("\n", $content);
        $mailMessage = (new MailMessage())->markdown(
            'emails.notification',
            ['locale' => $locale]
        )->model($this->reservation)->subject(config('notify.subject_prefix') . $subject);

        if ($this->withIcs) {
            $this->attachIcs($mailMessage, $data);
        }

        foreach ($contents as $content) {
            $mailMessage->line($content);
        }

        if ($notifiable instanceof AnonymousNotifiable) {
            $mailMessage->action('', '');
        } else {
            $mailMessage->action(Lang::get($buttonText, [], $locale), $this->getUrl());
        }

        return $mailMessage;
    }

    protected function setLocale(string $locale): void
    {
        $locale = Str::lower($locale);
        $fullLocale = Str::lower($locale) . '_' . Str::upper($locale) . '.utf8';
        Lang::setLocale($locale);
        setlocale(LC_TIME, $locale, $fullLocale);
        Carbon::setLocale($locale);
        Carbon::setUtf8(false);
    }

    protected function getTimeLapseForHumans(Carbon $startDate, Carbon $endDate): string
    {
        $startDate = get_date_in_user_timezone($startDate) ?? $startDate;
        $endDate = get_date_in_user_timezone($endDate) ?? $endDate;
        if ($startDate->isSameDay($endDate)) {
            return $startDate->formatLocalized("%A %d %B %Y %R") . ' - ' . $endDate->format('H:i');
        }

        return $startDate->formatLocalized("%A %d %B %Y %R") . ' - ' . $endDate->formatLocalized("%A %d %B %Y %R");
    }

    protected function getEntity(?Content $reservation = null): Content
    {
        if (is_null($reservation)) {
            $reservation = $this->reservation;
        }
        $type = Str::before(class_to_type($reservation), '_reservation');
        if ($reservation->{Str::camel($type)} === null || $reservation->{Str::camel($type)}->isEmpty()) {
            $class = type_to_class($type);
            /** @var Content $entity */
            $entity = new $class();
            $entity->title = 'Unkown';

            return $entity;
        }

        return $reservation->{Str::camel($type)}->first();
    }

    protected function getLanguageContent(string $key, array $data, string $locale): string
    {
        $type = Str::before(class_to_type($this->reservation), '_reservation');
        $translationKey = 'notifications.reservation.email.' . $type . '.' . $this->notificationName . '.' . $key;

        if (($translated = Lang::get($translationKey, $data, $locale)) != $translated) {
            return $translationKey;
        }

        return Lang::get(
            'notifications.reservation.email.' . $this->notificationName . '.' . $key,
            $data,
            $locale
        );
    }

    protected function attachIcs(MailMessage $mailMessage, array $data): void
    {
        if (is_null($this->reservation->created_at) || is_null($this->reservation->updated_at)) {
            return;
        }

        $startDate = get_date($this->reservation->start_date);
        $endDate = get_date($this->reservation->end_date);
        $created = get_date($this->reservation->created_at);
        $updated = get_date($this->reservation->updated_at);
        if (!$startDate || !$endDate || !$created || !$updated) {
            Log::info('[Attach ICS] There is a problem with given dates !');
            return;
        }

        $ics = [
            'BEGIN:VCALENDAR',
            'PRODID:-//Inside//Inside Reservation//EN',
            'VERSION:2.0',
            'CALSCALE:GREGORIAN',
            'METHOD:REQUEST',
            'BEGIN:VEVENT',
            'LOCATION:' . $this->getEntity()->title == null ? '' : $this->getEntity()->title,
            'DTSTART:' . $startDate->format('Ymd\THis\Z'),
            'DTEND:' . $endDate->format('Ymd\THis\Z'),
            'DTSTAMP:' . $created->format('Ymd\THis\Z'),
            #  'ORGANIZER;SENT-BY="MAILTO:' . $this->model->authors->email . '":MAILTO:'.config('mail.from.address', 'organizer@example.com'),
            'ORGANIZER:MAILTO:' . (($data['to.uuid'] === $data['organizer.uuid']) ? config(
                'mail.from.address',
                'organizer@example.com'
            ) : $data['organizer.email']),
            'UID:' . $this->reservation->uuid,
            'SEQUENCE:' . time(),
            'DESCRIPTION:' . $this->reservation->body,
            'CREATED:' . $created->format('Ymd\THis\Z'),
            'LAST-MODIFIED:' . $updated->format('Ymd\THis\Z'),
            'SUMMARY:' . $data['entity.title'] . ' - ' . $this->reservation->title,
        ];
        if ($this->reservation->frequency !== null) {
            $ics[] = $this->reservation->frequency;
            if ($this->reservation->exceptions !== null) {
                $exceptions = explode(',', $this->reservation->exceptions);
                $icsExceptions = [];
                foreach ($exceptions as $exception) {
                    try {
                        $icsExceptions[] = Carbon::createFromTimeString($exception, 'UTC')->format('Ymd\THis\Z');
                    } catch (\Throwable $e) {
                    }
                }
                if (!empty($icsExceptions)) {
                    $ics[] = 'EXDATE:' . implode(',', $icsExceptions);
                }
            }
        }
        if ($this->reservation->internalUsers) {
            foreach ($this->reservation->internalUsers as $p) {
                $ics[] = $this->getIcsAttendee($this->getFullName($p), $p->email);
            }
        }
        if ($this->reservation->external_users !== null) {
            foreach (explode(',', $this->reservation->external_users) as $p) {
                $email = filter_var(trim($p), FILTER_VALIDATE_EMAIL);
                if ($email) {
                    $ics[] = $this->getIcsAttendee($email, $email);
                }
            }
        }

        $ics = array_merge(
            $ics,
            [
                'END:VEVENT',
                'END:VCALENDAR',
            ]
        );

        $icsLines = [];
        foreach ($ics as $line) {
            foreach ($this->fold($line) as $l) {
                $icsLines[] = $l;
            }
        }

        // Attach to the mail
        $mailMessage->attachData(
            implode("\r\n", $icsLines),
            'invite.ics',
            ['mime' => 'text/calendar; charset=UTF-8; method=REQUEST', 'as' => 'invite.ics']
        );
    }

    private function getIcsAttendee(string $name, string $email): string
    {
        return 'ATTENDEE;CUTYPE=INDIVIDUAL;ROLE=REQ-PARTICIPANT;PARTSTAT=NEEDS-ACTION=;RSVP=TRUE;CN=' . $name
            . ';X-NUM-GUESTS=0:mailto:' . $email;
    }

    private function getFullName(Users $user): string
    {
        if (is_string($user->firstname) && !empty($user->firstname) && is_string($user->lastname)
            && !empty($user->lastname)
        ) {
            return Str::ucfirst(Str::lower($user->firstname)) . ' ' . Str::upper($user->lastname);
        } elseif (is_string($user->firstname) && !empty($user->firstname)) {
            return Str::ucfirst(Str::lower($user->firstname));
        } elseif (is_string($user->lastname) && !empty($user->lastname)) {
            return Str::upper($user->lastname);
        } else {
            return 'Unknown';
        }
    }

    private function fold(string $string): array
    {
        $lines = [];
        if (function_exists('mb_strcut')) {
            while (strlen($string) > 0) {
                if (strlen($string) > 75) {
                    $lines[] = mb_strcut($string, 0, 75, 'utf-8');
                    $string  = ' ' . mb_strcut($string, 75, strlen($string), 'utf-8');
                } else {
                    $lines[] = $string;
                    $string  = '';
                    break;
                }
            }
        } else {
            $array = preg_split('/(?<!^)(?!$)/u', $string);
            $line = '';
            $lineNo = 0;
            if (is_array($array)) {
                foreach ($array as $char) {
                    $charLen = strlen($char);
                    $lineLen = strlen($line);
                    if ($lineLen + $charLen > 75) {
                        $line = ' ' . $char;
                        ++$lineNo;
                    } else {
                        $line .= $char;
                    }
                    $lines[$lineNo] = $line;
                }
            }
        }

        return $lines;
    }

    protected function getUrl(bool $absolute = true): string
    {
        $baseUrl = config('app.url') . '/';

        return ($absolute ? $baseUrl : '') . ($this->customSlug === null ? $this->getEntity()->slug[0]
                : $this->customSlug) . '?date=' . $this->reservation->start_date . '&id=' . $this->reservation->uuid;
    }

    public function toDatabase(User $notifiable): array
    {
        /** @var NotificationType $notificationType */
        $notificationType = NotificationType::where('action', 'reservation')->firstOrFail();
        $startDate = get_date($this->reservation->start_date);
        $endDate = get_date($this->reservation->end_date);
        if (!$startDate || !$endDate) {
            throw new \Exception('There is a problem with given dates !');
        }
        $locale = $notifiable->langcode ?? config('app.locale');
        $this->setLocale($locale);

        return [
            'notification_type_id' => $notificationType->id,
            'type'                 => $notificationType->type,
            'parent'               => null,
            'user_uuid'            => $notifiable->uuid,
            'notifiable_type'      => get_class($this->reservation),
            'notifiable_uuid'      => $this->reservation->uuid,
            'notifiable_langcode'  => $notifiable->langcode,
            'data'                 => [
                'notification_name' => $this->notificationName,
                'timelapse'         => $this->getTimeLapseForHumans($startDate, $endDate),
                'entities'          => $this->getEntity()->toArray(),
                'from'              => $this->user->uuid,
                'url'               => $this->getUrl(false),
            ],
        ];
    }
}
