<?php

namespace Inside\Reservation\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentSynchedEvent;
use Inside\Content\Facades\Schema;
use Inside\Reservation\Events\ParticipantsOnBookingModificationWarned;
use Inside\Reservation\Notifications\ParticipationRemovedNotification;
use Inside\Reservation\Notifications\ReservationCreatedNotification;
use Inside\Reservation\Notifications\ReservationEntityChangedNotification;

class WarnParticipantOnBookingRelationChanges implements ShouldQueue
{
    /**
     * @param ContentSynchedEvent $event
     * @return void
     */
    public function handle(ContentSynchedEvent $event): void
    {
        $reservation = $event->model;
        $type = Str::before(class_to_type($reservation), '_reservation');
        $notified = collect();

        if (is_null($reservation->creator)) {
            return;
        }

        if (
            "{$type}_reservations" === class_to_type($reservation) && Schema::hasContentType($type)
            && Schema::hasContentType("{$type}_categories")
            && Schema::hasContentType("{$type}_reservations")
            && $reservation->exists
        ) {
            if ($event->fieldName === 'internal_users') {
                $addedParticipant   = $event->dirty->diff($event->original);
                $removedParticipant = $event->original->diff($event->dirty);
                foreach ($addedParticipant as $user) {
                    /** @var ?User $user */
                    $user = User::find($user->uuid);
                    if (!is_null($user) && $notified->contains($user->email)) {
                        continue;
                    }
                    if ($user) {
                        Notification::send(
                            $user,
                            (new ReservationCreatedNotification(
                                $reservation,
                                $reservation->creator,
                                env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                            ))->withIcs()
                        );

                        $notified->push($user->email);
                    }
                }
                foreach ($removedParticipant as $user) {
                    /** @var User $user */
                    $user = User::find($user->uuid);
                    if ($notified->contains($user->email)) {
                        continue;
                    }
                    Notification::send(
                        $user,
                        new ParticipationRemovedNotification(
                            $reservation,
                            $reservation->creator,
                            env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                        )
                    );
                    $notified->push($user->email);
                }
            } elseif ($event->fieldName === $type) {
                if ($reservation->internalUsers !== null && config('reservation.warn.modification.internal', true)) {
                    foreach ($reservation->internalUsers as $user) {
                        if ($notified->contains($user->email)) {
                            continue;
                        }
                        /** @var User $user */
                        $user = User::find($user->uuid);
                        Notification::send(
                            $user,
                            (new ReservationEntityChangedNotification(
                                $reservation,
                                $reservation->creator,
                                env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                            ))->withIcs()
                        );
                        $notified->push($user->email);
                    }
                }

                if ($reservation->external_users !== null && config('reservation.warn.modification.external', true)) {
                    $externalUsers = explode(',', $reservation->external_users);
                    foreach ($externalUsers as $email) {
                        $email = filter_var(trim($email), FILTER_VALIDATE_EMAIL);
                        if ($email === false || $notified->contains($email)) {
                            continue;
                        }
                        Notification::route('mail', $email)->notify(
                            (new ReservationEntityChangedNotification(
                                $reservation,
                                $reservation->creator,
                                env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                            ))->withIcs()
                        );
                        $notified->push($email);
                    }
                }
            }

            ParticipantsOnBookingModificationWarned::dispatch($reservation, $notified->toArray());
        }
    }
}
