<?php

namespace Inside\Reservation\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Facades\Schema;
use Inside\Reservation\Events\ParticipantsOnBookingCreationWarned;
use Inside\Reservation\Notifications\ReservationCreatedNotification;

class WarnParticipantOnBookingCreated implements ShouldQueue
{
    /**
     * @param ContentCreatedEvent $event
     * @return void
     */
    public function handle(ContentCreatedEvent $event): void
    {
        $reservation = $event->model;
        $type = Str::before(class_to_type($reservation), '_reservation');

        if (is_null($reservation->creator)) {
            return;
        }

        if (
            "{$type}_reservations" === class_to_type($reservation) && Schema::hasContentType($type)
            && Schema::hasContentType("{$type}_categories")
            && Schema::hasContentType("{$type}_reservations")
        ) {
            $notified = collect();
            if ($reservation->internalUsers !== null && config('reservation.warn.creation.internal', true)) {
                foreach ($reservation->internalUsers as $user) {
                    if ($notified->contains($user->email)) {
                        continue;
                    }

                    /** @var User $user */
                    $user = User::find($user->uuid);
                    Notification::send(
                        $user,
                        (new ReservationCreatedNotification(
                            $reservation,
                            $reservation->creator,
                            env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                        ))->withIcs()
                    );
                    $notified->push($user->email);
                }
            }

            if ($reservation->external_users !== null && config('reservation.warn.creation.external', true)) {
                $externalUsers = explode(',', $reservation->external_users);
                foreach ($externalUsers as $email) {
                    $email = filter_var(trim($email), FILTER_VALIDATE_EMAIL);
                    if ($email === false || $notified->contains($email)) {
                        continue;
                    }
                    Notification::route('mail', $email)->notify(
                        (new ReservationCreatedNotification(
                            $reservation,
                            $reservation->creator,
                            env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                        ))->withIcs()
                    );
                    $notified->push($email);
                }
            }

            if (
                $reservation->creator !== null
                && config('reservation.warn.creation.author', false)
                && !$notified->contains($reservation->creator->email)
            ) {
                Notification::send(
                    $reservation->creator,
                    (new ReservationCreatedNotification(
                        $reservation,
                        $reservation->creator,
                        env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                    ))->withIcs()
                );
                $notified->push($reservation->creator->email);
            }

            if (
                $reservation->organizer !== null
                && config('reservation.warn.creation.organizer', true)
                && $reservation->organizer->isNotEmpty()
                && !$notified->contains($reservation->organizer->first()->email)
            ) {
                /** @var ?User $user */
                $user = User::find($reservation->organizer->first()->uuid);
                if ($user) {
                    Notification::send(
                        $user,
                        (new ReservationCreatedNotification(
                            $reservation,
                            $reservation->creator,
                            env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                        ))->withIcs()
                    );
                    $notified->push($user->email);
                }
            }

            ParticipantsOnBookingCreationWarned::dispatch($reservation, $notified->toArray());
        }
    }
}
