<?php

namespace Inside\Reservation\Repositories;

use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use Inside\Database\Eloquent\Builder;

class ReservationRepository
{
    /**
     * @param string $type
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @param array $filters
     * @return Builder
     */
    public function getReservationListQuery(
        string $type,
        Carbon $startDate,
        Carbon $endDate,
        array $filters = []
    ): Builder {
        $query = call_user_func(type_to_class("{$type}_reservations") . '::query');
        $table = type_to_table("{$type}_reservations");
        $query->where(
            function ($query) use ($startDate, $endDate, $table) {
                $query->whereNotNull($table . '.frequency')->orWhereBetween(
                    $table . '.start_date',
                    [$startDate, $endDate]
                )->orWhereBetween($table . '.end_date', [$startDate, $endDate])->orWhere(
                    function ($query) use ($startDate, $endDate, $table) {
                        $query->whereDate($table . '.start_date', '<=', $startDate)->whereDate(
                            $table . '.end_date',
                            '>=',
                            $endDate
                        );
                    }
                );
            }
        );

        if (isset($filters[$type]['uuid'])) {
            $restriction = $filters[$type]['uuid'];
            if (is_string($restriction) && !empty($restriction)) {
                $restriction = [$restriction];
            }

            if (is_array($restriction)) {
                $query->whereHas(
                    Str::camel($type),
                    function ($query) use ($restriction, $type) {
                        $query->whereIn(type_to_table($type) . '.uuid', $restriction);
                    }
                );
            }
        }
        $query->where($table . '.status', true);
        $customFilters = config('reservation.index.custom_filters', false);
        if ($customFilters && is_callable($customFilters)) {
            $customFilters($filters, $type, $query);
        } elseif (is_array($customFilters)) {
            foreach ($customFilters as $customFilter) {
                if ($customFilter && is_callable($customFilter)) {
                    $customFilter($filters, $type, $query);
                }
            }
        }

        return $query;
    }
}
