<?php

namespace Inside\Reservation\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentUpdatingEvent;
use Inside\Content\Facades\Schema;
use Inside\Reservation\Events\ParticipantsOnBookingModificationWarned;
use Inside\Reservation\Notifications\ParticipationRemovedNotification;
use Inside\Reservation\Notifications\ReservationCancelledNotification;
use Inside\Reservation\Notifications\ReservationCreatedNotification;
use Inside\Reservation\Notifications\ReservationDatesChangedNotification;

class WarnParticipantOnBookingUpdating implements ShouldQueue
{
    /**
     * @param ContentUpdatingEvent $event
     * @return void
     */
    public function handle(ContentUpdatingEvent $event): void
    {
        $reservation = $event->model;
        $type = Str::before(class_to_type($reservation), '_reservation');
        $warned = false;
        $notified = false;

        if (is_null($reservation->creator)) {
            return;
        }

        if (
            "{$type}_reservations" === class_to_type($reservation) && Schema::hasContentType($type)
            && Schema::hasContentType("{$type}_categories")
            && Schema::hasContentType("{$type}_reservations")
        ) {
            if (! $reservation->status && $reservation->isDirty('status')) {
                $notified = collect();

                if ($reservation->internalUsers !== null && config('reservation.warn.modification.internal', true)) {
                    foreach ($reservation->internalUsers as $user) {
                        if ($notified->contains($user->email)) {
                            continue;
                        }
                        /** @var User $user */
                        $user = User::find($user->uuid);
                        Notification::send(
                            $user,
                            new ReservationCancelledNotification(
                                $reservation,
                                $reservation->creator,
                                env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                            )
                        );
                        $notified->push($user->email);
                    }
                }

                if ($reservation->external_users !== null && config('reservation.warn.modification.external', true)) {
                    $externalUsers = explode(',', $reservation->external_users);
                    foreach ($externalUsers as $email) {
                        $email = filter_var(trim($email), FILTER_VALIDATE_EMAIL);
                        if ($email === false || $notified->contains($email)) {
                            continue;
                        }
                        Notification::route('mail', $email)->notify(
                            new ReservationCancelledNotification(
                                $reservation,
                                $reservation->creator,
                                env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                            )
                        );
                        $notified->push($email);
                    }
                }
                $warned = true;
            } elseif ($reservation->isDirty('start_date') || $reservation->isDirty('end_date')) {
                $notified = collect();
                if ($reservation->internalUsers !== null && config('reservation.warn.modification.internal', true)) {
                    foreach ($reservation->internalUsers as $user) {
                        if ($notified->contains($user->email)) {
                            continue;
                        }
                        /** @var ?User $user */
                        $user = User::find($user->uuid);
                        if ($user) {
                            Notification::send(
                                $user,
                                (new ReservationDatesChangedNotification(
                                    $reservation,
                                    $reservation->creator,
                                    env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                                ))->withIcs()
                            );
                            $notified->push($user->email);
                        }
                    }
                }

                if ($reservation->external_users !== null && config('reservation.warn.modification.external', true)) {
                    $externalUsers = explode(',', $reservation->external_users);
                    foreach ($externalUsers as $email) {
                        $email = filter_var(trim($email), FILTER_VALIDATE_EMAIL);
                        if ($email === false || $notified->contains($email)) {
                            continue;
                        }
                        Notification::route('mail', $email)->notify(
                            (new ReservationDatesChangedNotification(
                                $reservation,
                                $reservation->creator,
                                env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                            ))->withIcs()
                        );
                        $notified->push($email);
                    }
                }
                $warned = true;
            } elseif ($reservation->isDirty('external_users')) {
                $addedParticipant = collect(explode(',', $reservation->getOriginal('external_users')));
                $removedParticipant = collect(explode(',', $reservation->external_users));
                foreach ($addedParticipant as $email) {
                    Notification::route('mail', $email)->notify(
                        (new ReservationCreatedNotification(
                            $reservation,
                            $reservation->creator,
                            env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                        ))->withIcs()
                    );
                }
                foreach ($removedParticipant as $email) {
                    Notification::route('mail', $email)->notify(
                        (new ParticipationRemovedNotification(
                            $reservation,
                            $reservation->creator,
                            env('RESERVATION_OVERRIDE_NOTIFICATION_SLUG')
                        ))
                    );
                }
            }

            if ($warned && $notified) {
                ParticipantsOnBookingModificationWarned::dispatch($reservation, $notified->toArray());
            }
        }
    }
}
