<?php

namespace Inside\Reservation\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;
use mysql_xdevapi\Exception;

class CreateContentTypes extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:reservation:create-content-types';

    /**
     * @var string
     */
    protected $signature = 'inside:reservation:create-content-types {resource?} {--default-min-time=} {--default-max-time=}';

    /**
     * @var string
     */
    protected $description = 'Create the content types needed for reservations';

    /**
     * @return void
     */
    public function handle(): void
    {
        $name = $this->argument('resource');
        $defaultMinTime = (int) $this->option('default-min-time');
        $defaultMaxTime = (int) $this->option('default-max-time');

        if (! is_string($name)) {
            throw new Exception('Error: $name is not an string type');
        }

        $contentTypeTitles = [
            $name => [],
            $name.'_categories' => [],
            $name.'_category' => [],
            $name.'_reservations' => [],
        ];

        foreach (list_languages() as $language) {
            foreach ($contentTypeTitles as $key => &$translations) {
                switch ($key) {
                    case $name:
                        $translations[$language] = Str::ucfirst($name);
                        break;
                    case $name.'_categories':
                        $translations[$language] = Str::ucfirst(Str::singular($name)).' types';
                        break;
                    case $name.'_category':
                        $translations[$language] = Str::ucfirst(Str::singular($name)).' type';
                        break;
                    case $name.'_reservations':
                        $translations[$language] = Str::ucfirst(Str::singular($name)).' reservations';
                        break;
                }
            }
        }

        $bridge = new BridgeContentType($this->getOutput());
        $fields = [
            'body' => [],
            'image' => [],
            'weight' => [],
            'users' => [],
            'start_date' => [],
            'end_date' => [],
            'default_cooldown' => [],
            'reservation_description' => [],
            'cooldown' => [],
            'frequency' => [],
            'internal_users' => [],
            'external_users' => [],
            'organizer' => [],
            'color' => [],
            'min_time' => [],
            'max_time' => [],
            'exceptions' => [],
            'location' => [],
            'equipments' => [],
            'capacity' => [],
            'description' => [],
        ];

        foreach ($fields as $field => &$translations) {
            foreach (list_languages() as $language) {
                $translations[$language] = Lang::get('reservation.fields.'.$field, [], $language);
            }
        }

        $contentTypes = [
            'categories' => [
                'options' => [
                    'translatable' => false,
                    'searchable' => false,
                    'global_searchable' => false,
                    'aliasable' => true,
                    'permissible' => true,
                    'listing_type' => 'category',
                    'categorizable' => true,
                    'name' => $name.'_categories',
                    'title' => $contentTypeTitles[$name.'_categories'],
                ],
                'fields' => [
                    [
                        'name' => 'color',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => '',
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 4,
                        'title' => $fields['color'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::STRING_WIDGET,
                            'settings' => [
                                'size' => 60,
                                'placeholder' => '',
                            ],
                            'weight' => 4,
                        ],
                        'type' => BridgeContentType::TEXT_FIELD,
                        'settings' => [
                            'max_length' => 255,
                            'is_ascii' => false,
                            'case_sensitive' => false,
                            'cardinality' => 1,
                        ],
                    ],
                ],
            ],
            'contents' => [
                'options' => [
                    'translatable' => false,
                    'searchable' => false,
                    'global_searchable' => false,
                    'aliasable' => true,
                    'permissible' => true,
                    'listing_type' => 'content',
                    'categorizable' => false,
                    'name' => $name,
                    'title' => $contentTypeTitles[$name],
                ],
                'fields' => [
                    [
                        'name' => 'body',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => '',
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 1,
                        'title' => $fields['body'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::WYSIWYG_WIDGET,
                            'settings' => [
                                'rows' => 9,
                                'summary_rows' => 3,
                                'placeholder' => '',
                            ],
                            'weight' => 14,
                        ],
                        'type' => BridgeContentType::WYSIWYG_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => $name.'_category',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => null,
                        'search_result_field' => false,
                        'required' => true,
                        'weight' => 0,
                        'title' => $contentTypeTitles[$name.'_category'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::SELECT_WIDGET,
                        ],
                        'type' => BridgeContentType::REFERENCE_FIELD,
                        'settings' => [
                            'target_type' => [$name.'_categories'],
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'image',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => false,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 2,
                        'title' => $fields['image'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::IMAGE_WIDGET,
                            'settings' => [
                                'progress_indicator' => 'throbber',
                                'preview_image_style' => 'thumbnail',
                            ],
                            'weight' => 1,
                        ],
                        'type' => BridgeContentType::IMAGE_FIELD,
                        'settings' => [
                            'uri_scheme' => 'public',
                            'default_image' => [
                                'uuid' => '',
                                'alt' => '',
                                'title' => '',
                                'width' => null,
                                'height' => null,
                            ],
                            'target_type' => 'file',
                            'display_field' => false,
                            'display_default' => false,
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'min_time',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => $defaultMinTime,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 2,
                        'title' => $fields['min_time'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::NUMBER_WIDGET,
                            'settings' => [],
                        ],
                        'type' => BridgeContentType::INTEGER_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'max_time',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => $defaultMaxTime,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 2,
                        'title' => $fields['max_time'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::NUMBER_WIDGET,
                            'settings' => [],
                        ],
                        'type' => BridgeContentType::INTEGER_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'weight',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => 0,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 3,
                        'title' => $fields['weight'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::NUMBER_WIDGET,
                            'settings' => [
                                'placeholder' => '',
                            ],
                            'weight' => 7,
                        ],
                        'type' => BridgeContentType::INTEGER_FIELD,
                        'settings' => [
                            'unsigned' => false,
                            'size' => 'normal',
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'default_cooldown',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => ['value' => 0],
                        'search_result_field' => false,
                        'required' => true,
                        'weight' => 3,
                        'title' => $fields['default_cooldown'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::SELECT_WIDGET,
                            'settings' => [],
                            'weight' => 8,
                        ],
                        'type' => BridgeContentType::LIST_INTEGER_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                            "allowed_values" => [
                                0,
                                5,
                                10,
                                15,
                                20,
                                25,
                                30,
                                35,
                                40,
                                45,
                                50,
                                55,
                                60,
                            ],
                        ],
                    ],
                    [
                        'name' => 'location',
                        'searchable' => false,
                        'translatable' => false,
                        'search_result_field' => false,
                        'searchable_filter' => false,
                        'required' => false,
                        'weight' => 9,
                        'title' => $fields['location'],
                        'widget' => [
                            'type' => BridgeContentType::STRING_WIDGET,
                            'settings' => [
                                'size' => 60,
                                'placeholder' => '',
                            ],
                            'weight' => 9,
                        ],
                        'type' => BridgeContentType::TEXT_FIELD,
                        'settings' => [
                            'max_length' => 255,
                            'is_ascii' => false,
                            'case_sensitive' => false,
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'equipments',
                        'searchable' => false,
                        'translatable' => false,
                        'search_result_field' => false,
                        'searchable_filter' => false,
                        'required' => false,
                        'weight' => 10,
                        'title' => $fields['equipments'],
                        'widget' => [
                            'type' => BridgeContentType::STRING_WIDGET,
                            'weight' => 10,
                            'settings' => [
                                'size' => 60,
                                'placeholder' => '',
                            ],
                        ],
                        'type' => BridgeContentType::TEXT_FIELD,
                        'settings' => [
                            'max_length' => 255,
                            'is_ascii' => false,
                            'case_sensitive' => false,
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'capacity',
                        'searchable' => false,
                        'translatable' => false,
                        'search_result_field' => false,
                        'searchable_filter' => false,
                        'required' => false,
                        'weight' => 11,
                        'title' => $fields['capacity'],
                        'widget' => [
                            'type' => BridgeContentType::STRING_WIDGET,
                            'settings' => [
                                'size' => 60,
                                'placeholder' => '',
                            ],
                            'weight' => 11,
                        ],
                        'type' => BridgeContentType::TEXT_FIELD,
                        'settings' => [
                            'max_length' => 255,
                            'is_ascii' => false,
                            'case_sensitive' => false,
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'description',
                        'searchable' => false,
                        'translatable' => false,
                        'search_result_field' => false,
                        'searchable_filter' => false,
                        'required' => false,
                        'weight' => 12,
                        'title' => $fields['description'],
                        'widget' => [
                            'type' => BridgeContentType::LONGTEXT_WIDGET,
                            'settings' => [
                                'rows' => 5,
                                'placeholder' => '',
                            ],
                            'weight' => 12,
                        ],
                        'type' => BridgeContentType::TEXTAREA_FIELD,
                        'settings' => [
                            'case_sensitive' => false,
                            'cardinality' => 1,
                        ],
                    ],
                ],
            ],
            'reservations' => [
                'options' => [
                    'translatable' => false,
                    'searchable' => false,
                    'global_searchable' => false,
                    'aliasable' => false,
                    'permissible' => true,
                    'listing_type' => 'content',
                    'categorizable' => false,
                    'name' => $name.'_reservations',
                    'title' => $contentTypeTitles[$name.'_reservations'],
                ],
                'fields' => [
                    [
                        'name' => $name,
                        'searchable' => false,
                        'translatable' => false,
                        'default' => null,
                        'search_result_field' => false,
                        'required' => true,
                        'weight' => 1,
                        'title' => $contentTypeTitles[$name],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::SELECT_WIDGET,
                            'weight' => 1,
                        ],
                        'type' => BridgeContentType::REFERENCE_FIELD,
                        'settings' => [
                            'target_type' => [$name],
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'start_date',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => false,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 1,
                        'title' => $fields['start_date'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::TIMESTAMP_WIDGET,
                            'settings' => [],
                            'weight' => 1,
                        ],
                        'type' => BridgeContentType::TIMESTAMP_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'frequency',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => '',
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 4,
                        'title' => $fields['frequency'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::STRING_WIDGET,
                            'settings' => [
                                'size' => 60,
                                'placeholder' => '',
                            ],
                            'weight' => 5,
                        ],
                        'type' => BridgeContentType::TEXT_FIELD,
                        'settings' => [
                            'max_length' => 255,
                            'is_ascii' => false,
                            'case_sensitive' => false,
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'end_date',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => false,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 1,
                        'title' => $fields['end_date'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::TIMESTAMP_WIDGET,
                            'settings' => [],
                            'weight' => 1,
                        ],
                        'type' => BridgeContentType::TIMESTAMP_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'cooldown',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => ['value' => 0],
                        'search_result_field' => false,
                        'required' => true,
                        'weight' => 3,
                        'title' => $fields['cooldown'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::SELECT_WIDGET,
                            'settings' => [],
                            'weight' => 8,
                        ],
                        'type' => BridgeContentType::LIST_INTEGER_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                            "allowed_values" => [
                                0,
                                5,
                                10,
                                15,
                                20,
                                25,
                                30,
                                35,
                                40,
                                45,
                                50,
                                55,
                                60,
                            ],
                        ],
                    ],
                    [
                        'name' => 'reservation_description',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => '',
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 1,
                        'title' => $fields['reservation_description'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::LONGTEXT_WIDGET,
                            'settings' => [
                                'rows' => 5,
                                'placeholder' => '',
                            ],
                            'weight' => 14,
                        ],
                        'type' => BridgeContentType::TEXTAREA_FIELD,
                        'settings' => [
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'internal_users',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => false,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 4,
                        'title' => $fields['internal_users'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::AUTOCOMPLETE_WIDGET,
                            'settings' => [
                                'match_operator' => 'CONTAINS',
                                'size' => 60,
                                'placeholder' => '',
                            ],
                            'weight' => 4,
                        ],
                        'type' => BridgeContentType::REFERENCE_FIELD,
                        'settings' => [
                            'target_type' => 'users',
                            'cardinality' => -1,
                        ],
                    ],
                    [
                        'name' => 'external_users',
                        'searchable' => false,
                        'translatable' => false,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 3,
                        'title' => $fields['external_users'],
                        'description' => [],
                        'default' => '',
                        'widget' => [
                            'type' => BridgeContentType::LONGTEXT_WIDGET,
                            'settings' => [
                                'rows' => 5,
                                'placeholder' => '',
                            ],
                            'weight' => 3,
                        ],
                        'type' => BridgeContentType::TEXTAREA_FIELD,
                        'settings' => [
                            'case_sensitive' => false,
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'organizer',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => false,
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 4,
                        'title' => $fields['organizer'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::AUTOCOMPLETE_WIDGET,
                            'settings' => [
                                'match_operator' => 'CONTAINS',
                                'size' => 60,
                                'placeholder' => '',
                            ],
                            'weight' => 2,
                        ],
                        'type' => BridgeContentType::REFERENCE_FIELD,
                        'settings' => [
                            'target_type' => 'users',
                            'cardinality' => 1,
                        ],
                    ],
                    [
                        'name' => 'exceptions',
                        'searchable' => false,
                        'translatable' => false,
                        'default' => '',
                        'search_result_field' => false,
                        'required' => false,
                        'weight' => 1,
                        'title' => $fields['exceptions'],
                        'description' => [],
                        'widget' => [
                            'type' => BridgeContentType::LONGTEXT_WIDGET,
                            'weight' => 1,
                        ],
                        'type' => BridgeContentType::TEXTAREA_FIELD,
                        'settings' => [],
                    ],
                ],
            ],
        ];

        foreach ($contentTypes as $contentType) {
            if (! Schema::hasModel($contentType['options']['name'])) {
                $this->info('Lancement de la création du type '.$contentType['options']['name']);
            } else {
                $this->info('Lancement de la mise-à-jour du type '.$contentType['options']['name']);
            }

            $bridge->contentTypeUpdateOrCreate(
                $contentType['options']['name'],
                $contentType['options'],
                $contentType['fields']
            );
        }
    }
}
