<?php

namespace Inside\Quiz\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside\Quiz\Contracts\Quiz as QuizContract;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Content\Models\Contents\Quiz;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Sections\Answer;
use Inside\Content\Models\Sections\Question;

class QuizService implements QuizContract
{
    /**
     * @param User $user
     * @param Quiz $quiz
     * @return int
     */
    public function userLastSubmission(User $user, Quiz $quiz): int
    {
        $submission = DB::table('inside_quiz_answers')->where(['user_uuid' => $user->uuid, 'quiz_uuid' => $quiz->uuid])->orderByDesc('submission')->value('submission');
        $submission = $submission ?? 1;

        if (!$quiz->quiz_unique_submission && $this->userAnswersCompleted($user, $submission, $quiz)) {
            $submission++;
        }

        return $submission;
    }

    /**
     * @param User $user
     * @param int $submission
     * @param Quiz $quiz
     * @return bool
     */
    public function userAnswersCompleted(User $user, int $submission, Quiz $quiz): bool
    {
        $questionCount = $quiz->sectionContent->count();
        $userQuestionCount = DB::table('inside_quiz_answers')->where(['user_uuid' => $user->uuid, 'quiz_uuid' => $quiz->uuid, 'submission' => $submission])->count();

        return $questionCount === $userQuestionCount;
    }

    /**
     * @param User $user
     * @param int $submission
     * @param Quiz $quiz
     * @param Question $question
     * @return bool
     */
    public function userAnswerExists(User $user, int $submission, Quiz $quiz, Question $question): bool
    {
        return DB::table('inside_quiz_answers')->where(['user_uuid' => $user->uuid, 'submission' => $submission, 'quiz_uuid' => $quiz->uuid, 'question_uuid' => $question->uuid])->exists();
    }

    /**
     * @param User $user
     * @param int $submission
     * @param Quiz $quiz
     * @return int
     */
    public function getUserScore(User $user, int $submission, Quiz $quiz): int
    {
        return DB::table('inside_quiz_answers')->where(['user_uuid' => $user->uuid, 'submission' => $submission, 'quiz_uuid' => $quiz->uuid, 'correct' => true])->count();
    }

    /**
     * @param Quiz $quiz
     * @return float
     */
    public function getAverage(Quiz $quiz): float
    {
        return (float) DB::table('inside_quiz_scores')->select(DB::raw("AVG(score) AS avg_score"))->where('quiz_uuid', $quiz->uuid)->groupBy('quiz_uuid')->value('avg_score');
    }

    /**
     * @param User $user
     * @param int $submission
     * @param Quiz $quiz
     * @param Question $question
     * @param array $answers
     * @param bool $correct
     * @return void
     */
    public function storeAnswer(User $user, int $submission, Quiz $quiz, Question $question, array $answers, bool $correct): void
    {
        DB::table('inside_quiz_answers')->insert(
            [
                'submission' => $submission,
                'user_uuid' => $user->uuid,
                'quiz_uuid' => $quiz->uuid,
                'question_uuid' => $question->uuid,
                'answer' => json_encode($answers),
                'correct' => $correct,
                'answered_at' => now()
            ]
        );
    }

    /**
     * @param User $user
     * @param int $submission
     * @param Quiz $quiz
     * @param int $score
     * @return void
     */
    public function storeScore(User $user, int $submission, Quiz $quiz, int $score): void
    {
        DB::table('inside_quiz_scores')->updateOrInsert(
            [
                'user_uuid' => $user->uuid,
                'quiz_uuid' => $quiz->uuid,
            ],
            [
                'submission' => $submission,
                'score' => $score
            ]
        );
    }

    /**
     * @param Quiz $quiz
     * @param User|null $user
     * @return array
     */
    public function extractQuestions(Quiz $quiz, User $user = null): array
    {
        $formatted = [];
        $questions = $quiz->sectionContent;

        foreach ($questions as $question) {
            $questionData = [
                'uuid' => $question->uuid,
                'title' => $question->question_title,
                'details' => $question->question_details,
                'image' => null,
                'answers' => [],
                'correct_answers_number' => 0,
            ];

            if ($user) {
                $questionData['answered'] = $this->userAnswerExists($user, $this->userLastSubmission($user, $quiz), $quiz, $question);
            }

            if (!empty($question->question_image)) {
                $questionData['image'] = $this->addStyledImageToQuestions($question);
            }

            $answers = $question->sectionContent;

            foreach ($answers as $answer) {
                $questionData['answers'][] = [
                    'uuid' => $answer->uuid,
                    'title' => $answer->answer_title,
                ];

                if ($answer->answer_correct) {
                    $questionData['correct_answers_number']++;
                }
            }

            $formatted[] = $questionData;
        }

        return $formatted;
    }

    /**
     * @param mixed $question
     *
     * @return array
     */
    protected function addStyledImageToQuestions($question): array
    {
        $image = [
            'main' => protected_file_url($question, 'question_image'),
        ];

        $options = Schema::getFieldOptions(class_to_type($question), 'question_image');
        if (isset($options['image_styles']) && !empty($options['image_styles'])) {
            $styles = ImageStyles::find($options['image_styles']);
            foreach ($styles as $style) {
                $image[Str::slug($style->title)] = protected_file_url($question, 'question_image', true, $style->title);
            }
        }

        return $image;
    }

    /**
     * @param Quiz $quiz
     * @return int
     */
    public function getNumberOfParticipants(Quiz $quiz): int
    {
        return DB::table('inside_quiz_scores')->where(['quiz_uuid' => $quiz->uuid])->count();
    }

    /**
     * @param Quiz $quiz
     * @return array
     */
    public function getResults(Quiz $quiz): array
    {
        $scores = DB::table('inside_quiz_scores')->where(['quiz_uuid' => $quiz->uuid])->get();
        $results = [];
        foreach ($scores as $score) {
            $answers = DB::table('inside_quiz_answers')->where(['quiz_uuid' => $quiz->uuid, 'user_uuid' => $score->user_uuid, 'submission' => $score->submission])->orderBy('answered_at')->get();
            $result = [
                'timestamp' => $answers->last()->answered_at
            ];

            if (!$quiz->is_anonymous) {
                $user = Users::find($score->user_uuid);
                $resultAnonymous = [
                    'Nom' => $user->lastname,
                    'Prénom' => $user->firstname,
                    'Score' => $score->score,
                    'submission' => $score->submission
                ];
                $result = array_merge($result, $resultAnonymous);
            }

            foreach ($answers as $answer) {
                $uuids = json_decode($answer->answer);
                $textAnswers = Answer::query()->whereIn('uuid', $uuids)->pluck('answer_title')->toArray();
                $result[$answer->question_uuid] = implode(', ', $textAnswers);
            }
            $results[] = $result;
        }

        return $results;
    }
}
