<?php

namespace Inside\Quiz\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\Contents\Quiz;
use Inside\Jobs\Bus\Dispatchable;
use Inside\Quiz\Events\QuizCSVGeneratedEvent;
use Inside\Quiz\Facades\Quiz as QuizService;
use League\Csv\CannotInsertRecord;
use League\Csv\Exception;
use League\Csv\Writer;

class GenerateQuizCSV implements ShouldQueue
{
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;
    use Dispatchable;

    protected const CSV_DOWNLOAD_FOLDER = 'csv/download';

    /**
     * @param string $userUuid
     * @param Quiz $quiz
     */
    public function __construct(
        protected string $userUuid,
        protected Quiz $quiz
    ) {
    }

    /**
     * @throws CannotInsertRecord
     * @throws Exception
     */
    public function handle(): void
    {
        Storage::disk('protected')->makeDirectory(self::CSV_DOWNLOAD_FOLDER);
        $filePath = self::CSV_DOWNLOAD_FOLDER.'/'.$this->userUuid.'.csv';
        $questions = QuizService::extractQuestions($this->quiz);

        $header = [
            'timestamp' => Lang::get('quiz.export.header.date')
        ];

        if (!$this->quiz->is_anonymous) {
            $headers = [
                'Nom' => 'Nom',
                'Prénom' => 'Prénom',
                'Score' => 'Score',
                'submission' => 'submission'
            ];
            $header = array_merge($header, $headers);
        }

        foreach ($questions as $question) {
            $header[$question['uuid']] = $question['title'];
        }

        $results = QuizService::getResults($this->quiz);
        $csv = Writer::createFromPath(Storage::disk('protected')->path($filePath), "w+")
            ->setDelimiter(';');
        $csv->insertOne($header);
        $csv->insertAll($results);
        QuizCSVGeneratedEvent::dispatch($this->userUuid, $filePath);
        $this->delete();
    }
}
