<?php

namespace Inside\Onboardings\Services;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Facades\ContentHelper;
use Inside\Database\Eloquent\Builder;
use Inside\Onboardings\Models\OnboardingGames;
use Inside\Onboardings\Models\OnboardingPieces;
use Inside\Onboardings\Models\OnboardingTurns;
use Inside\Support\Str;
use Inside\Validation\Rule;
use Inside\Validation\ValidateRequests;

class GameService
{
    use ValidateRequests;

    protected function saveImage(string $image): string
    {
        // handle image (move it from chunks)
        $folder = 'onboarding-games';
        $newPath = $folder.'/'.basename($image);
        $storage = Storage::disk('protected');
        if (!$storage->exists($folder)) {
            $storage->makeDirectory($folder);
        }
        if ($storage->exists($newPath)) {
            $storage->delete($newPath);
        }
        if (!Storage::disk('local')->copy($image, 'protected/'.$newPath)) {
            throw new \Exception('un souci est survenu sur la sauvegarde du fichier');
        }
        return $newPath;
    }

    protected function checkDuplicates(Collection $datas): void
    {
        $duplicates = $datas->duplicates('position');

        if ($duplicates->isNotEmpty()) {
            throw new \Exception("Positions should be unique");
        }
    }

    public function createGame(array $datas): array
    {
        $datas['image'] = $this->saveImage($datas['image']);
        if (isset($datas['default_image'])) {
            $datas['default_image'] = $this->saveImage($datas['default_image']);
        }
        $datas['langcode'] = Auth::user()->langcode;

        // need to handle pieces too;
        $pieces = $datas['pieces'];
        unset($datas['pieces']);

        $this->checkDuplicates(collect($pieces));

        // todo rework this later
        $game = OnboardingGames::create($datas);
        $datas['id'] = $game->id;
        $datas['image'] = 'protected_files/'.$game->image;
        if (isset($datas['default_image'])) {
            $datas['default_image'] = 'protected_files/'.$game->default_image;
        }

        foreach ($pieces as $piece) {
            $piece['game_id'] = $game->id;
            $piece['urls'] = json_encode($piece['urls']);
            OnboardingPieces::create($piece);
        }

        return array_merge($datas, ['pieces' => $pieces]);
    }

    public function editGame(OnboardingGames $game, array $datas): array
    {
        // need to handle images properly ...
        // then we assume it's a new file
        if (Str::startsWith($datas['image'], 'chunks/')) {
            $datas['image'] = $this->saveImage($datas['image']);
        } else {
            $datas['image'] = str_replace('protected_files/', '', $datas['image']);
        }
        if (isset($datas['default_image']) && Str::startsWith($datas['default_image'], 'chunks/')) {
            $datas['default_image'] = $this->saveImage($datas['default_image']);
        } else {
            $datas['default_image'] = str_replace('protected_files/', '', $datas['default_image']);
        }

        $id = $game->id;
        $pieces = $datas['pieces'];
        unset($datas['pieces']);
        $datas['id'] = $id;

        //check if there is only one position per game
        $this->checkDuplicates(collect($pieces));
        $this->validateData($pieces, [
            'position' => Rule::unique('inside_onboardings_pieces')->where(function ($query) use ($id) {
                return $query->where('game_id', $id);
            }),
        ]);

        $game->update($datas);

        foreach ($pieces as $piece) {
            $existing_piece = OnboardingPieces::query()
                ->where('position', $piece['position'])
                ->where('game_id', $id)
                ->first();
            $piece['urls'] = json_encode($piece['urls']);
            if ($existing_piece instanceof OnboardingPieces) {
                $existing_piece->update($piece);
            } elseif ($piece['position'] >= 0 && $piece['position'] < $game->pieces_count) {
                // a new piece to insert
                $piece['game_id'] = $id;
                OnboardingPieces::create($piece);
            } else {
                // should not happen
                throw new \Exception("error: Your piece should have a position superior or equal to 0 and inferior to ".$game->pieces_count);
            }
        }

        $all_pieces = OnboardingPieces::query()->where('game_id', $id)->select(['position', 'title', 'urls', 'hover', 'image'])->get()->toArray();

        $game_fields = OnboardingGames::query()->where('id', $id)->first()->toArray();
        $game_fields['image'] = 'protected_files/'.$game_fields['image'];
        if (isset($game_fields['default_image']) && !empty($game_fields['default_image'])) {
            $game_fields['default_image'] = 'protected_files/'.$game_fields['default_image'];
        }
        return array_merge($game_fields, ['pieces' => $all_pieces]);
    }

    /**
     * @param string $id
     * @param string|array $filters
     * @return mixed
     */
    public function getParticipants(string $id, $filters)
    {
        $query = OnboardingTurns::with('information')
            ->where('game_id', $id)
            ->addSelect([
                'last_update' => OnboardingTurns::from('inside_onboarding_turns as play')
                    ->select(['play.created_at'])
                    ->whereColumn('play.game_id', 'inside_onboarding_turns.game_id')
                    ->whereColumn('play.uuid', 'inside_onboarding_turns.uuid')
                    ->orderBy('created_at', 'desc')
                    ->take(1),
                'count' => OnboardingTurns::from('inside_onboarding_turns as player')
                    ->selectRaw('count(*)')
                    ->whereColumn('player.game_id', 'inside_onboarding_turns.game_id')
                    ->whereColumn('player.uuid', 'inside_onboarding_turns.uuid')
                    ->take(1),
                'pieces_count' => OnboardingGames::from('inside_onboarding_games as game')
                    ->select('pieces_count')
                    ->where('id', $id),
            ])
            ->withCasts(['last_update' => 'timestamp'])
            ->groupBy('uuid');

        $query = $this->filter($query, $filters);

        return ContentHelper::getResultFromQuery($query, [], true);
    }

    /**
     * @param OnboardingGames $game
     * @param string $path
     * @return array|null
     */
    public function hasFoundPiece(OnboardingGames $game, string $path)
    {
        $user = Auth::user();
        $piece = OnboardingPieces::query()->where('game_id', $game->id)->where('urls', $path)->first();
        if (!$piece instanceof OnboardingPieces) {
            // no piece exist with that url
            return null;
        }
        $query = OnboardingTurns::query()->where('game_id', $game->id)->where('uuid', $user->uuid);
        $turn = $query->where('piece_position', $piece->position)->where('url', $path)->first();
        $found = $query->count();
        if ($turn) {
            // user already found that piece
            return null;
        }

        return [
            "image" => $piece->image,
            "found" => $found,
            "total" => $game->pieces_count,
        ];
    }

    /**
     * @param Builder $query
     * @param string|array $filters
     * @return mixed
     */
    public function filter(Builder $query, $filters)
    {
        if (is_string($filters)) {
            $filters = json_decode($filters, true) ?? [];
        }
        if (!isset($filters['sort'])) {
            return $query;
        }
        $sort = $filters['sort'];
        [$column, $filter] = explode(':', $sort);
        if (!in_array($filter, ['asc', 'desc'])) {
            return $query;
        }
        // only these 4 fields can be filtered on
        if (in_array($column, ['email', 'lastname'])) {
            $query = $query->join('inside_content_users', 'inside_onboarding_turns.uuid', '=', 'inside_content_users.uuid');
            return $query->orderBy('inside_content_users.'.$column, $filter);
        }
        if ($column === "updated_at") {
            return $query->orderBy('last_update', $filter);
        }
        if ($column === "count") {
            return $query->orderBy($column, $filter);
        }
        return $query;
    }
}
