<?php

namespace Inside\Onboardings\Http\Controllers;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Onboardings\Facades\Game;
use Inside\Onboardings\Models\OnboardingGames;
use Inside\Onboardings\Models\OnboardingPieces;
use Inside\Onboardings\Models\OnboardingTurns;
use Inside\Validation\ValidateRequests;
use Laravel\Lumen\Routing\Controller;

class GameController extends Controller
{
    use ValidateRequests;

    protected const RULES = [
        'title' => 'required|string',
        'description' => 'string|nullable',
        'end_description' => 'string|nullable',
        'piece_height' => 'required|numeric',
        'piece_width' => 'required|numeric',
        'image_dimensions' => 'required|string',
        'image' => 'required|string',
        'default_image' => 'string',
        'pieces' => 'required',
        'pieces.*.position' => 'required|numeric',
        'pieces.*.urls' => 'required',
        'pieces.*.title' => 'required|string',
        'pieces.*.hover' => 'required|string',
        'pieces.*.image' => 'required|string',
    ];

    /**
     * @param Request $request
     * @return array
     * @throws \Illuminate\Validation\ValidationException
     */
    public function create(Request $request): array
    {
        $additional_rules = [
            'status' => 'required|boolean',
            'pieces_count' => 'required|numeric',
            'columns' => 'required|numeric',
            'rows' => 'required|numeric',
        ];
        // add first set of validation for the pieces (would need one more after)
        $datas = $this->validateData($request->all(), array_merge(self::RULES, $additional_rules));

        return Game::createGame($datas);
    }

    public function edit(Request $request, string $id): array
    {
        $game = OnboardingGames::findOrFail($id);
        $datas = $this->validateData($request->all(), self::RULES);
        return Game::editGame($game, $datas);
    }

    public function toggleStatus(Request $request, string $id): array
    {
        $game = OnboardingGames::findOrFail($id);
        $game->status = !$game->status;
        $game->save();
        return [
            "status" => $game->status,
        ];
    }

    public function delete(Request $request, string $id): array
    {
        $game = OnboardingGames::findOrFail($id);
        $game->delete();
        return [
            "success" => true,
        ];
    }

    public function piecesFound(Request $request): Collection
    {
        $user = Auth::user();
        $games = OnboardingGames::select('id')
            ->with([
                'turns' => function ($query) use ($user) {
                    $query->select('game_id', 'piece_id')
                        ->where('uuid', $user->uuid);
                },
            ])->get();
        // not needed for now, but keep it just in case
        /*addSelect([
                     'count' => OnboardingTurns::from('inside_onboarding_turns as turn')
                         ->selectRaw('count(*)')
                         ->whereColumn('inside_onboarding_games.id', 'turn.game_id')
                         ->where('turn.uuid', $user->uuid)
                         ->take(1),
                    'last_found_id' => OnboardingTurns::from('inside_onboarding_turns as turns')
                        ->select('id')
                        ->whereColumn('turns.game_id', 'inside_onboarding_games.id')
                        ->where('turns.uuid', $user->uuid)
                        ->take(1),
            ])*/

        return $games;
    }

    public function getAllGames(Request $request): array
    {
        // return all games and all pieces
        $user = Auth::user();
        $games = OnboardingGames::with('pieces')
            ->where('langcode', $user->langcode)
            ->get();

        // ugly, but we'll change it later
        $games = $games->each(function ($game) {
            $game->image = 'protected_files/' . $game->image;
            if ($game->default_image) {
                $game->default_image = 'protected_files/' . $game->default_image;
            }
        });

        return $games->toArray();
    }

    public function getGame(Request $request, string $id): OnboardingGames
    {
        /** @var OnboardingGames $game */
        $game = OnboardingGames::with('pieces')->findOrFail($id);
        $game->image = 'protected_files/' . $game->image;
        if ($game->default_image) {
            $game->default_image = 'protected_files/' . $game->default_image;
        }
        return $game;
    }

    public function getStatistics(Request $request, string $id): array
    {
        $game = OnboardingGames::findOrFail($id);

        $pieces = OnboardingPieces::query()->where('game_id', $id)->get();
        $datas = [];

        /** @var OnboardingPieces $piece */
        foreach ($pieces as $piece) {
            $datas[] = [
              "id" => $piece->id,
              "title" => $piece->title,
              "urls" => json_decode($piece->urls),
              "users" => OnboardingTurns::query()->where('game_id', $id)->where('piece_position', $piece->position)->count(),
            ];
        }
        return [
            'data' => $datas,
        ];
    }

    public function getParticipants(Request $request, string $id): array
    {
        $game = OnboardingGames::findOrFail($id);

        return [
            'data' => Game::getParticipants($id, $request->get('filters', [])),
        ];
    }

    public function getProgression(Request $request): array
    {
        $games = $this->piecesFound($request);
        $datas = [];
        foreach ($games as $game) {
            $datas[$game->id] = $game->turns->pluck('piece_id');
        }
        return $datas;
    }
}
