<?php

namespace Inside\Newsletters\Views\Components;

use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\View;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Newsletters\Facades\Permission;
use Inside\Support\Str;
use Inside\Support\View\Component;

class Section extends Component
{
    /** @var string */
    public $name;

    /** @var array */
    public $options;

    /** @var mixed */
    public $fieldsPrefix;

    /** @var \Illuminate\Support\Collection */
    public $items;

    /** @var Content */
    public $newsletter;

    /** @var string */
    public $locale;

    /** @var string */
    public $type;

    public ?User $user;

    public function __construct(string $name, string $type, array $options = [])
    {
        /** @var ?User */
        $user = User::find(View::shared('user')?->uuid);

        $this->name = $name;
        $this->type = $type;
        $this->options = $options;
        $this->fieldsPrefix = View::shared('fieldsPrefix');
        $this->newsletter = View::shared('newsletter');
        $this->locale = View::shared('locale');
        $this->user = $user;
        $this->items = collect($this->options['items'] ?? []);
    }

    public function render()
    {
        $orientation = $this->options['orientation'] ?? 'vertical';
        if (! in_array($orientation, ['vertical', 'horizontal'])) {
            $orientation = 'vertical';
        }
        if ($this->items->isEmpty()) {
            // Generate $items content using $fields of $this->newsletter
            // Convention ==> field Name = type name
            $fields = $this->options['types'] ?? ['news'];
            try {
                foreach ($fields as $fieldKey => $field) {
                    $fieldKey = is_string($fieldKey) ? $fieldKey : $field;
                    $field = $this->fieldsPrefix.$fieldKey;
                    if (
                        InsideSchema::hasField(class_to_type($this->newsletter), $field)
                        && $this->newsletter->{Str::camel($field)}->isNotEmpty()
                    ) {
                        $items = collect(ContentHelper::getContentsInLocale($this->newsletter->{Str::camel($field)}, $this->locale))
                            ->filter(fn (Content $content) => Permission::hasAccess($content, $this->user));

                        $this->items = $this->items->merge($items);
                    }
                }
            } catch (Exception $e) {
                Log::error($e->getMessage());
            }
        }
        $sectionTitleStrKey = $this->options['sectionTitleStrKey'] ?? 'newsletter.'.$this->name.'_title';
        $sectionLinkStrKey = $this->options['sectionLinkStrKey'] ?? 'newsletter.'.$this->name.'_link';
        $sectionMarginBottom = in_array($this->type, ['sops', 'rops', 'updated_rops', 'updated_sops']) ? 50 : 0;

        $sorts = $this->options['sorts'] ?? [];
        foreach ($sorts as $sort) {
            $dir = 'asc';
            if (strpos($sort, ':')) {
                [$sort, $dir] = explode(':', $sort);
            }
            if ($dir == 'asc') {
                $this->items = $this->items->sortBy($sort);
            } else {
                $this->items = $this->items->sortByDesc($sort);
            }
        }
        $sectionComponentName = 'iln::partials.'.$this->type;
        $columnWidthPercent = null;
        if ($orientation == 'horizontal') {
            $columnWidthPercent = (int) (100 / $this->items->count());
        }

        return view(
            'iln::components.section',
            compact('sectionTitleStrKey', 'sectionLinkStrKey', 'sectionComponentName', 'orientation', 'columnWidthPercent', 'sectionMarginBottom')
        );
    }
}
