<?php

namespace Inside\Newsletters\Views\Components;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\View;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Kernel\Filesystem\Facades\FileStreamer;
use Inside\Support\View\Component;
use Inside\User\Models\User;

class Banner extends Component
{
    private const DEFAULT_LANGCODE = 'fr';

    /** @var array */
    public $options;

    /** @var \Illuminate\Support\Collection */
    public $items;

    /** @var Content */
    public $newsletter;

    /** @var string */
    public $locale;

    /** @var User */
    public $user;

    public function __construct(array $options = [])
    {
        $this->options = $options;
        $this->newsletter = View::shared('newsletter');
        $this->locale = View::shared('locale');
        $this->items = collect($this->options['items'] ?? []);
        $this->user = View::shared('user');
    }

    public function render()
    {
        $options = $this->options;
        $locale = $this->locale;
        $newsletter = $this->newsletter;
        $user = $this->user;

        $titleStrKey = $options['titleStrKey'] ?? 'newsletter.title';
        $titleColor = $options['titleColor'] ?? '#fff';
        $subTitleStrKey = $options['subTitleStrKey'] ?? 'newsletter.sub_title';
        $subTitleColor = $options['subTitleColor'] ?? '#fff';
        $backgroundColor = $options['backgroundColor'] ?? '#1F1635';

        $backgroundImageUrl = $this->backgroundImageUrl($options);
        $backgroundImageAsCover = $options['backgroundImageAsCover'] ?? false;
        $bannerImageTextColor = $options['bannerImageTextColor'] ?? '#000';
        $bannerWidth = $options['bannerWidth'] ?? 600;
        $bannerTitle = $options['title'] ?? '';
        $bannerSubTitle = $options['subTitle'] ?? '';

        $useTitleField = $options['useTitleField'] ?? false;
        $titleFieldName = $options['titleFieldName'] ?? 'banner_title';
        $subTitleFieldName = $options['subTitleFieldName'] ?? 'banner_subtitle';

        $useImageField = $options['useImageField'] ?? false;
        $bannerImageField = $options['bannerImageField'] ?? 'banner_image';

        $languages = list_languages();

        // Title
        $title = __($titleStrKey);
        $subTitle = __($subTitleStrKey);

        if ($useTitleField) {
            // Title
            if (InsideSchema::hasField(class_to_type($newsletter), $titleFieldName . '_' . $locale) &&
                !empty($newsletter->{$titleFieldName . '_' . $locale})
            ) {
                $title = $newsletter->{$titleFieldName . '_' . $locale};
            } elseif (InsideSchema::hasField(class_to_type($newsletter), $titleFieldName)
                && empty($newsletter->{$titleFieldName})
            ) {
                $title = $newsletter->{$titleFieldName};
            }

            // SubTitle
            if (InsideSchema::hasField(class_to_type($newsletter), $subTitleFieldName . '_' . $locale) &&
                !empty($newsletter->{$subTitleFieldName . '_' . $locale})
            ) {
                $title = $newsletter->{$subTitleFieldName . '_' . $locale};
            } elseif (InsideSchema::hasField(class_to_type($newsletter), $subTitleFieldName) &&
                empty($newsletter->{$subTitleFieldName})
            ) {
                $title = $newsletter->{$subTitleFieldName};
            }
        }

        // Check if newsletter has field
        $localStorage = Storage::disk('local');

        if ($backgroundImageUrl == $this->newsletter->uuid) {
            $backgroundImageUrl = url('/newsletter_assets', [$backgroundImageUrl,'nl_banner']);
        } elseif ($useImageField) {
            if (InsideSchema::hasField(class_to_type($newsletter), $bannerImageField . '_' . $locale) &&
                !empty($newsletter->{$bannerImageField . '_' . $locale}) &&
                $localStorage->exists($newsletter->{$bannerImageField . '_' . $locale})
            ) {
                $backgroundImageUrl = $newsletter->{$bannerImageField . '_' . $locale};
            } elseif (InsideSchema::hasField(class_to_type($newsletter), $bannerImageField) &&
                !empty($newsletter->{$bannerImageField}) &&
                $localStorage->exists($newsletter->{$bannerImageField})
            ) {
                $backgroundImageUrl = $newsletter->{$bannerImageField};
            }
            if (!is_null($backgroundImageUrl)) {
                $backgroundImageUrl = newsletter_signed_url(
                    $user,
                    $newsletter,
                    $newsletter,
                    $backgroundImageUrl,
                    $this->backgroundImageUrl($options)
                );
            }
        } elseif (!is_null($backgroundImageUrl)) {
            /** @var array $pathInfo */
            $pathInfo = pathinfo($backgroundImageUrl);
            $localeBackgroundImageUrl = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '_' . $locale . '.' . $pathInfo['extension'];
            $localeBackgroundImagePath = cms_base_path($localeBackgroundImageUrl);
            if (!File::exists(cms_base_path($localeBackgroundImageUrl)) || count($languages) == 1) {
                $backgroundImageUrl = url(path: $backgroundImageUrl, secure: true);
            } else {
                $backgroundImageUrl = url($localeBackgroundImageUrl);
            }
        }

        return view(
            'iln::components.banner',
            compact(
                'backgroundImageUrl',
                'backgroundImageAsCover',
                'bannerWidth',
                'backgroundColor',
                'title',
                'subTitle',
                'titleColor',
                'subTitleColor',
                'bannerImageTextColor',
                'bannerTitle',
                'bannerSubTitle'
            )
        );
    }

    private function backgroundImageUrl(array $options): ?string
    {
        if (empty($options['backgroundImageUrl'])) {
            return null;
        }
        $langcode = $this->user->langcode;
        return is_array($options['backgroundImageUrl']) ?
            ($options['backgroundImageUrl'][$langcode] ?? $options['backgroundImageUrl'][self::DEFAULT_LANGCODE]) : $options['backgroundImageUrl'];
    }
}
