<?php

namespace Inside\Newsletters\Services;

use Carbon\Carbon;
use Inside\Content\Models\Contents\Users;
use Inside\Newsletters\Models\NewslettersSent;
use Inside\Content\Facades\ContentHelper;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;
use Inside\Newsletters\Models\NewslettersStatisticsGeneric;
use Inside\Permission\Models\Role;

class StatisticsService
{
    public function getNewslettersStatisticsQuery(array $filters = [], ?string $search = null): Builder
    {
        $fromDate = $this->extractDateFromFilters($filters, 'from');
        $toDate = $fromDate === null ? null : $this->extractDateFromFilters($filters, 'to');
        $langcode = $filters['langcode'] ?? null;
        if (!isset($filters['type'])) {
            $config = config('newsletters');
            $filters['type'] = Arr::first(array_keys($config['types']));
        }
        $newsletterStatsTable = type_to_stats_table($filters['type']);
        $newsletterTable = type_to_table($filters['type']);

        return NewslettersStatisticsGeneric::bind($filters['type'])
                    ->rightJoin(DB::raw("$newsletterTable as newsletter"), "$newsletterStatsTable.content_uuid", "=", "newsletter.uuid")
                    ->leftJoin(DB::raw("
                            (
                                SELECT
                                    newsletter_sent.uuid,
                                    COUNT(DISTINCT newsletter_sent.user_uuid) AS total_sent
                                FROM inside_newsletters_sent AS newsletter_sent
                                GROUP BY newsletter_sent.uuid
                            ) AS sent_stats
                        "), "sent_stats.uuid", "=", DB::raw("COALESCE($newsletterStatsTable.content_uuid, newsletter.uuid)"))
                    ->leftJoin(DB::raw("
                            (
                                SELECT
                                    content_uuid,
                                    COUNT(DISTINCT CASE WHEN statistic_type = 'viewed' THEN user_uuid END) AS unique_views,
                                    COUNT(CASE WHEN statistic_type = 'viewed' THEN 1 END) AS total_views,
                                    COUNT(DISTINCT CASE WHEN statistic_type = 'clicked' THEN user_uuid END) AS unique_clicks,
                                    COUNT(CASE WHEN statistic_type = 'clicked' THEN 1 END) AS clicks
                                FROM $newsletterStatsTable
                                GROUP BY content_uuid
                            ) AS stats
                        "), "stats.content_uuid", "=", DB::raw("COALESCE($newsletterStatsTable.content_uuid, newsletter.uuid)"))
                    ->select(DB::raw("
                        COALESCE($newsletterStatsTable.content_uuid, newsletter.uuid) AS content_uuid,
                        COALESCE(sent_stats.total_sent, 0) AS total_sent,
                        COALESCE(stats.unique_views, 0) AS unique_views,
                        COALESCE(stats.total_views, 0) AS total_views,
                        COALESCE(stats.unique_clicks, 0) AS unique_clicks,
                        COALESCE(stats.clicks, 0) AS clicks,
                        newsletter.title,
                        newsletter.created_at,
                        newsletter.langcode,
                        newsletter.sent_at
                    "))
                    ->groupBy(DB::raw("COALESCE($newsletterStatsTable.content_uuid, newsletter.uuid),
                               newsletter.title, newsletter.created_at, newsletter.langcode, newsletter.sent_at,
                               sent_stats.total_sent, stats.unique_views, stats.total_views, stats.unique_clicks, stats.clicks"))
                    ->when($search, fn ($query, $search) => $query->where('newsletter.title', 'LIKE', '%' . $search . '%'))
                    ->when($fromDate, fn ($query) => $query->whereDate('newsletter.sent_at', '>=', $fromDate?->endOfDay()->toDateString()))
                    ->when($toDate, fn ($query) => $query->whereDate('newsletter.sent_at', '<=', $toDate?->endOfDay()->toDateString()))
                    ->when($langcode && in_array($langcode, list_languages()), fn ($query) => $query->where('newsletter.langcode', '=', $langcode))
                    ->orderBy('newsletter.created_at', 'desc');
    }

    public function getNewslettersStatistics(array $filters = [], ?string $search = null): Paginator | Collection | LengthAwarePaginator | array
    {
        $query = $this->getNewslettersStatisticsQuery($filters, $search);
        $count = $this->getQueryCount($query);

        return ContentHelper::getResultFromQuery($query, $filters, true, $count);
    }

    public function getNewsletterViewsQuery(string $newsletterUuid, ?string $search = null, array $filters = []): Builder
    {
        $statType = $filters['event'] ?? 'viewed';
        $fromDate = $this->extractDateFromFilters($filters, 'from');
        $toDate = $fromDate === null ? null : $this->extractDateFromFilters($filters, 'to');
        $notViewed = in_array($filters['not_viewed'] ?? null, [true, 1, '1', 'true'], true);
        if (!isset($filters['type'])) {
            $config = config('newsletters');
            $filters['type'] = Arr::first(array_keys($config['types']));
        }
        $newsletterStatsTable = type_to_stats_table($filters['type']);

        if ($notViewed) {
            $roleIDs = json_decode(type_to_class($filters['type'])::find($newsletterUuid)->newsletter_roles);

            if (empty($roleIDs)) {
                /** @var Role $authenticatedRole */
                $authenticatedRole = Role::query()->where('name', 'authenticated')->first();
                $roleIDs = [$authenticatedRole->id];
            }

            $query = Users::leftJoin($newsletterStatsTable, function ($join) use ($newsletterUuid, $statType) {
                $join->on('user_uuid', '=', 'uuid')
                        ->where('content_uuid', $newsletterUuid)
                        ->where('statistic_type', '=', $statType);
            })
                ->join('inside_users_roles', 'uuid', '=', 'inside_users_roles.user_uuid')
                ->whereIn(
                    'inside_users_roles.role_id',
                    $roleIDs
                )
                ->join('inside_newsletters_sent', 'inside_newsletters_sent.user_uuid', '=', 'inside_content_users.uuid')
                ->where(
                    'inside_newsletters_sent.uuid',
                    $newsletterUuid
                )
                ->whereNull('content_uuid')
                ->distinct('email');
        } else {
            $query = Users::join($newsletterStatsTable, 'user_uuid', '=', 'uuid')
                ->where('statistic_type', $statType)
                ->where('content_uuid', $newsletterUuid)
                ->when($fromDate, fn (Builder $query, Carbon $fromDate) => $query->whereDate($newsletterStatsTable.'.created_at', '>=', $fromDate->setTime(0, 0)->toDateString()))
                ->when($toDate, fn (Builder $query, Carbon $toDate) => $query->whereDate($newsletterStatsTable.'.created_at', '<=', $toDate->setTime(23, 59, 59)->toDateString()))
                ->groupBy('user_uuid', 'uuid');
        }

        $query
            ->select([
                'inside_content_users.uuid',
                'email',
                'firstname',
                'lastname',
            ])
            ->selectRaw("CONCAT(firstname, ' ', lastname) AS full_name");

        if (! empty($search)) {
            $query->whereRawLike("CONCAT(firstname, ' ', lastname)", "%$search%");
        }

        return $query;
    }

    public function getNewsletterViews(string $alertUuid, ?string $search = null, array $filters = []): Paginator | Collection | LengthAwarePaginator | array
    {
        $query = $this->getNewsletterViewsQuery($alertUuid, $search, $filters);
        $count = $this->getQueryCount($query);
        return ContentHelper::getResultFromQuery($query, $filters, true, $count);
    }

    public function getQueryCount(Builder $query): int
    {
        return $query->get()->count();
    }

    private function extractDateFromFilters(array $filters, string $filterName): ?Carbon
    {
        if (! isset($filters[$filterName])) {
            return null;
        }

        try {
            $date = $filters[$filterName];
            return get_date($date, 'Y-m-d')?->setTime(0, 0);
        } catch (Throwable $exception) {
            Log::error("[Statistics::extractDateFromFilters] Failed to parse date from $date => {$exception->getMessage()}");
            return null;
        }
    }
}
