<?php

namespace Inside\Newsletters\Providers;

use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\View;
use Inside\Newsletters\Console\Commands\CreateScheduledNewsletterCommand;
use Inside\Newsletters\Console\Commands\SetupNewsletterSettings;
use Inside\Newsletters\Console\Commands\SyncNewsletterCommand;
use Inside\Newsletters\Console\Commands\SendScheduledNewsletterCommand;
use Inside\Newsletters\Console\Commands\CreateNewslettersStatsTables;
use Inside\Newsletters\Console\Commands\MigrateOldNewslettersStats;
use Inside\Newsletters\Events\NewsletterTestTriggeredEvent;
use Inside\Newsletters\Events\NewsletterTriggeredEvent;
use Inside\Newsletters\Facades\UrlSigner;
use Inside\Newsletters\Facades\Permission;
use Inside\Newsletters\Http\Middlewares\NewslettersFormMiddleware;
use Inside\Newsletters\Listeners\NewsletterTestTriggeredListener;
use Inside\Newsletters\Listeners\NewsletterTriggeredListener;
use Inside\Newsletters\Services\NewsletterSettingService;
use Inside\Newsletters\Services\UrlSignerService;
use Inside\Newsletters\Services\PermissionService;
use Inside\Support\EventServiceProvider;

/**
 * Newsletters service provider
 *
 * @category Class
 * @package  Inside\Newsletters\Listeners\NewslettersServiceProvider
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NewslettersServiceProvider extends EventServiceProvider
{
    protected array $listen = [
        NewsletterTriggeredEvent::class => [
            NewsletterTriggeredListener::class,
        ], NewsletterTestTriggeredEvent::class => [
            NewsletterTestTriggeredListener::class,
        ],
    ];

    protected array $commands = [
        SyncNewsletterCommand::class,
        CreateScheduledNewsletterCommand::class,
        SendScheduledNewsletterCommand::class,
        CreateNewslettersStatsTables::class,
        MigrateOldNewslettersStats::class
    ];

    public array $bindings = [
        UrlSigner::class => UrlSignerService::class,
        Permission::class => PermissionService::class
    ];

    public function register(): void
    {
        $this->mergeConfigFrom(
            __DIR__.'/../../config/newsletters.php',
            'newsletters'
        );

        $this->app->middleware([
            NewslettersFormMiddleware::class,
        ]);

        $this->app->singleton(NewsletterSettingService::class, function ($app) {
            return new NewsletterSettingService();
        });

        if ($this->app->runningInConsole()) {
            $this->app->singleton(
                'command.setup.settings',
                function ($app) {
                    return new SetupNewsletterSettings(
                        $app->make(NewsletterSettingService::class)
                    );
                }
            );
            $this->commands('command.setup.settings');
        }

        View::addNamespace(
            'iln',
            realpath(cms_base_path('vendor/maecia/inside-newsletters/resources/views/iln')) ?: ''
        );
        Blade::componentNamespace(
            'Inside\\Newsletters\\Views\\Components',
            'iln'
        );
    }
}
