<?php

namespace Inside\Newsletters\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContent;
use Inside\Newsletters\Events\NewsletterTriggeredEvent;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Models\Role;
use Inside\User\Models\User;

/**
 * Content created listener
 *
 * @category Class
 * @package  Inside\Newsletters\Listeners\NewsletterTriggeredListener
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NewsletterTriggeredListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(NewsletterTriggeredEvent $event): void
    {
        $model = $event->model;

        /** @var NotificationType|null $type */
        $type = NotificationType::where('event', NewsletterTriggeredEvent::class)->first();

        if (is_null($type)) {
            return;
        }

        $config = config('newsletters');

        // get role(s) from model
        $roleIDs = json_decode($model->newsletter_roles);

        if (empty($roleIDs)) {
            /** @var Role $authenticatedRole */
            $authenticatedRole = \Inside\Permission\Models\Role::query()->where('name', 'authenticated')->first();
            $roleIDs = [$authenticatedRole->id];
        }

        $query = User::query()->where('status', 1);
        $query->join('inside_users_roles', 'inside_users.uuid', '=', 'inside_users_roles.user_uuid')->whereIn(
            'inside_users_roles.role_id',
            $roleIDs
        );

        $query->whereNotIn('inside_users.uuid', function ($subQuery) use ($model) {
            $subQuery->select('user_uuid')->from('inside_newsletters_unsubscriptions')->where('type', class_to_type($model));
        });

        $subscribers = $query->get();

        if (config('louvrehotels') && class_exists('Inside\LHG\Facades\RightApi')) {
            $rightApiFacade = '\Inside\LHG\Facades\RightApi';
            $notifialedUsers = $rightApiFacade::getNotifiableUsers($rightApiFacade::NEWSLETTER);
            $subscribers = $subscribers->filter(fn ($subscriber) => $notifialedUsers->has($subscriber->email));
        }

        $subscribers->each(function ($subscriber) use ($type, $model) {
            $sent = DB::table('inside_newsletters_sent')->where('uuid', $model->uuid)->where('user_uuid', $subscriber->uuid)->first();

            if (! empty($sent)) {
                return;
            }

            $this->route = $subscriber;

            DB::table('inside_newsletters_sent')->insert([
                'uuid' => $model->uuid,
                'type' => get_class($model),
                'user_uuid' => $subscriber->uuid,
                'date_sent' => Carbon::now()->toDateString(),
                'langcode' => $model->langcode,
            ]);

            $this->notify($type, $subscriber, $model, $this->getNewsletterData($model), [], 'newsletter');
        });

        if ($subscribers->isNotEmpty() && empty($model->sent_at)) {
            $this->updateNewsletterSentAt($model);
        }
    }

    private function updateNewsletterSentAt(Content $model): void
    {
        (new BridgeContent())->contentUpdate(class_to_type($model::class), [
            'uuid' => $model->uuid,
            'sent_at' => Carbon::now('UTC')
        ]);
    }

    private function getNewsletterData(Content $model): array
    {
        $notificationConfig = config("newsletters.types.$model->content_type.notification", []);
        $data = config("newsletters.types.$model->content_type.data", []);

        $data = collect($data)
            ->map(fn (mixed $value) => is_callable($value) ? $value($model) : $value)
            ->filter()
            ->all();

        return array_merge($data, ['extra' => $notificationConfig]);
    }
}
