<?php

namespace Inside\Newsletters\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Inside\Notify\Listeners\BaseNotificationListener;
use Illuminate\Support\Facades\DB;
use Inside\Newsletters\Models\NewslettersSent;
use Carbon\Carbon;
use Inside\Settings\Models\Setting;
use Inside\Support\Str;
use Inside\Content\Facades\ContentHelper;
use Inside\Notify\Models\NotificationType;
use Inside\User\Models\User;
use Illuminate\Database\Eloquent\Model;
use Inside\Newsletters\Facades\Permission;

class SendScheduledNewsletter extends BaseNotificationListener implements ShouldQueue
{
    use InteractsWithQueue;
    use Queueable;

    private Model $model;
    private bool $isTest;
    private User $user;
    private NotificationType $notificationType;
    private array $dataNotification;

    /**
     * @param User $user
     * @param Model $model
     * @param NotificationType $notificationType
     * @param array $dataNotification
     * @param bool $isTest
     */
    public function __construct(User $user, Model $model, NotificationType $notificationType, array $dataNotification, bool $isTest = false)
    {
        $this->model = $model;
        $this->isTest = $isTest;
        $this->user = $user;
        $this->notificationType = $notificationType;
        $this->dataNotification = $dataNotification;
    }

    public function handle(): void
    {
        $isEmailSendingEnabled = (bool)Setting::where('key', 'email_sending_enabled')->first()?->value;
        if (!$isEmailSendingEnabled) {
            Log::warning('MailNotifications are disabled - newsletter send skipped ');
            return;
        }

        $model = $this->model;
        $fieldsOptions = config("newsletters.types.$model->contentType.scheduled_options.fields");
        $fields = array_keys($fieldsOptions);

        $items = collect($fields)->map(
            fn ($field) =>
            collect(
                ContentHelper::getContentsInLocale(
                    data: $model->{Str::camel($field)},
                    locale: $this->user->langcode ?? config('app.locale')
                )
            )->filter(fn ($content) => Permission::hasAccess($content, $this->user))
        )->flatten();

        if ($items->isEmpty()) {
            return;
        }

        $sent = NewslettersSent::where('uuid', $model->uuid)->where('user_uuid', $this->user->uuid)->first();

        if (!empty($sent)) {
            return;
        }

        $this->route = $this->user;

        if (!$this->isTest) {
            DB::table('inside_newsletters_sent')->insert([
                'uuid' => $model->uuid,
                'type' => get_class($model),
                'user_uuid' => $this->user->uuid,
                'date_sent' => Carbon::now()->toDateString(),
                'langcode' => $model->langcode,
            ]);
        }

        $this->notify($this->notificationType, $this->user, $model, [
            'extra' => $this->dataNotification,
            'test' => $this->isTest
        ], [], 'newsletter');
    }
}
