<?php

declare(strict_types=1);

namespace Inside\Newsletters\Jobs;

use Carbon\Carbon;
use Exception;
use Illuminate\Console\OutputStyle;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Lang;
use Inside\Content\Facades\Schema;
use Inside\Database\Eloquent\Builder;
use Inside\Host\Bridge\BridgeContent;
use Inside\Jobs\Job;
use Inside\Permission\Models\Role;
use Illuminate\Support\Facades\Log;

final class CreateScheduledNewsletter extends Job
{
    private const PUBLISHED_STATUS = 1;
    private ?BridgeContent $bridgeContent = null;

    public function __construct(
        private string       $newsletterType,
        private ?OutputStyle $outputStyle = null,
    ) {
    }

    public function handle(): void
    {
        if (!Schema::hasModel($this->newsletterType)) {
            throw new Exception("Invalid type $this->newsletterType");
        }

        if (!config("newsletters.types.$this->newsletterType.scheduled", false)) {
            return;
        }

        $options = config("newsletters.types.$this->newsletterType.scheduled_options") ?? [];

        if (empty($options)) {
            throw new Exception("Missing scheduled_options configurations for newsletter type $this->newsletterType");
        }

        $contents = $this->getFieldsContents($options);

        if (empty($contents)) {
            $this->outputStyle?->writeln("No content to create the newsletter <fg=yellow>$this->newsletterType</fg=yellow> ❌");

            return;
        }

        $data = collect()
            ->merge($this->getIntroduction($options))
            ->merge($contents)
            ->all();

        $titleKey = $options['titleKey'] ?? "newsletters.scheduled.$this->newsletterType.title";
        $titleDateFormat = $options['titleDateFormat'] ?? 'd/m/Y';

        $data['title'] = Lang::get($titleKey, ['model.published_at' => now()->format($titleDateFormat)]);

        $data['newsletter_roles'] = Role::whereIn('name', $options['roles'] ?? ['authenticated'])
            ->pluck('id')
            ->whenNotEmpty(fn (Collection $collection) => $collection->toJson(), fn () => null);

        $newsletterUuid = $this->getBridge()->contentInsert($this->newsletterType, $data);

        Log::info("[Newsletter] type : $this->newsletterType with uuid : $newsletterUuid created !");
        $this->outputStyle?->writeln("Newsletter <fg=yellow>$this->newsletterType</fg=yellow> created ✅");
    }

    private function getBridge(): BridgeContent
    {
        return $this->bridgeContent ??= new BridgeContent();
    }

    private function getIntroduction(array $options): array
    {
        $languages = collect(list_languages());

        $introductionField = $options['introductionFieldName'] ?? 'introduction';
        $introductionKey = $options['introductionKey'] ?? "newsletters.scheduled.$this->newsletterType.introduction";

        if ($languages->count() === 1) {
            return [$introductionField => Lang::get($introductionKey)];
        }

        return $languages
            ->mapWithKeys(fn (string $langcode) => ["{$introductionField}_$langcode" => Lang::get($introductionKey, [], $langcode)])
            ->all();
    }

    private function getFieldsContents(array $options): array
    {
        $dateLimit = !isset($options['date_limit']) ?
             type_to_class($this->newsletterType)::query()
                ->latest('published_at')
                ->first()?->published_at : (
                    empty($options['date_limit']) ? null : Carbon::now()->subDays($options['date_limit'])->getTimestamp()
                );

        return collect($options['fields'])
            ->mapWithKeys(function (mixed $fieldOptions, string $fieldName) use ($dateLimit) {
                $contentType = is_string($fieldOptions) ? $fieldOptions : ($fieldOptions['content_type'] ?? null);

                if (!is_string($contentType) || !Schema::hasModel($contentType)) {
                    throw new Exception("Invalid configurations for field $fieldName");
                }

                $query = type_to_class($contentType)::query()
                    ->where('langcode', config('app.locale'))
                    ->where('status', self::PUBLISHED_STATUS)
                    ->where('published_at', '<', now()->toDateTimeString())
                    ->when(
                        $dateLimit,
                        fn (Builder $query, int $dateLimit) => $query->where('published_at', '>', get_date($dateLimit))
                    );

                $customQuery = $fieldOptions['custom_query'] ?? null;

                if (is_callable($customQuery)) {
                    $query = $customQuery($query);
                }

                $contents = $query?->pluck('uuid')->all() ?? [];

                return [$fieldName => $contents];
            })
            ->filter()
            ->all();
    }
}
