<?php

declare(strict_types=1);

namespace Inside\Newsletters\Http\Controllers;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\WysiwygImage;
use Inside\Kernel\Exception\FileNotFoundException;
use Inside\Kernel\Filesystem\Facades\FileStreamer;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * @class NewslettersContentUrlController
 *
 * Allow streaming newsletter files.
 */
final class NewslettersContentUrlController
{
    private const WYSIWYG_IMAGE = 'wysiwyg_image';

    /**
     * @param string $contentType
     * @param string $contentUuid
     * @return StreamedResponse
     */
    public function serve(string $contentType, string $contentUuid): StreamedResponse
    {
        $defaultImage = config("newsletter.defaultImage") ?? cms_base_path() . '/vendor/maecia/inside-newsletters/resources/images/iln/default_thumbnail.png';
        try {
            if ($contentType === self::WYSIWYG_IMAGE) {
                $model = WysiwygImage::withoutGlobalScopes()->findOrFail($contentUuid);
            } else {
                $model = type_to_class($contentType)::withoutGlobalScopes()->findOrFail($contentUuid);
            }
        } catch (\Throwable $exception) {
            try {
                $model = section_type_to_class($contentType)::withoutGlobalScopes()->findOrFail($contentUuid);
            } catch (ModelNotFoundException $modelNotFoundException) {
                Log::warning("[Serve Newsletter] try to access to unknown $contentType uuid ($contentUuid)");
                throw $modelNotFoundException;
            }
        }

        if ($contentType === self::WYSIWYG_IMAGE) {
            $path = str_replace(storage_path('app/'), '', $model->path);
            return FileStreamer::streamFile($path);
        }

        $contentImage =
            str((string) get_stylized_image_url($model, 'image', 'image_card', true))
                ->after('storage/app/')
                ->toString();


        $disk = Storage::disk('local');

        if (! $disk->exists($contentImage)) {
            return FileStreamer::streamLocalFile($defaultImage);
        }

        return FileStreamer::streamFile($contentImage);
    }
}
