<?php

namespace Inside\Newsletters\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Inside\Newsletters\Events\NewsletterTriggeredEvent;
use Inside\Content\Facades\Schema;
use Inside\Newsletters\Models\NewslettersSent;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;
use Inside\Newsletters\Jobs\SendScheduledNewsletter;
use Inside\Settings\Models\Setting;
use Inside\User\Models\User;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Models\Role;
use Illuminate\Support\Facades\Queue;

class SendScheduledNewsletterCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:newsletter:scheduled:send';

    /**
     * @var string
     */
    protected $signature = 'inside:newsletter:scheduled:send {--type=} {--email=} {--test}';

    /**
     * @var string
     */
    protected $description = 'Send scheduled newsletter';

    public function handle(): void
    {
        $isEmailSendingEnabled = (bool)Setting::where('key', 'email_sending_enabled')->first()?->value;
        if (!$isEmailSendingEnabled) {
            Log::warning('MailNotifications are disabled - newsletter send skipped ');
            return;
        }
        /** @var NotificationType|null $notificationType */
        $notificationType = NotificationType::where('event', NewsletterTriggeredEvent::class)->first();

        if (is_null($notificationType)) {
            return;
        }

        $type = strval($this->option('type'));
        $email = $this->option('email');

        if (! Schema::hasModel($type)) {
            throw new Exception("Invalid type $type");
        }

        if (! config("newsletters.types.$type.scheduled", false)) {
            return;
        }

        $config = config('newsletters');
        $data = $config['types'][$type] ?? [];

        $newsletter = type_to_class($type)::whereDate('created_at', Carbon::today())
            ->latest('created_at')
            ->firstOrFail();

        if ($email) {
            $users = User::where('email', $email);
        } else {
            if ($this->option('test')) {
                $users = User::whereHas(
                    'roles',
                    fn ($query) => $query->whereIn('name', $data['scheduled_options']['test_roles'] ?? ['super_administrator'])
                )
                    ->where('status', 1)
                    ->get();
            } else {
                $roleIDs = json_decode($newsletter->newsletter_roles);
                if (empty($roleIDs)) {
                    /** @var Role $authenticatedRole */
                    $authenticatedRole = Role::query()->where('name', 'authenticated')->first();
                    $roleIDs = [$authenticatedRole->id];
                }

                $query = User::query()
                    ->where('status', 1)
                    ->join('inside_users_roles', 'inside_users.uuid', '=', 'inside_users_roles.user_uuid')
                    ->whereIn(
                        'inside_users_roles.role_id',
                        $roleIDs
                    )
                    ->whereNotIn('inside_users.uuid', function ($subQuery) use ($newsletter) {
                        $subQuery->select('user_uuid')->from('inside_newsletters_unsubscriptions')->where('type', class_to_type($newsletter));
                    });

                $users = $query->get();
            }
        }

        if (config('louvrehotels') && class_exists('Inside\LHG\Facades\RightApi')) {
            $rightApiFacade = '\Inside\LHG\Facades\RightApi';
            $notifialedUsers = $rightApiFacade::getNotifiableUsers($rightApiFacade::NEWSLETTER);
            $users = $users->filter(fn ($user) => $notifialedUsers->has($user->email));
        }

        $users->each(function ($user, $index) use ($notificationType, $newsletter, $data) {
            Queue::later($index, new SendScheduledNewsletter($user, $newsletter, $notificationType, $data['notification'], (bool)$this->option('test')));
        });
    }
}
