<?php

namespace Inside\Newsletters\Views\Components\Partials;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\View;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Facades\Package;
use Inside\Settings\Models\Setting;
use Inside\Support\Str;
use Inside\Support\View\Component;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;

class Documents extends Component
{
    /** @var array */
    public $itemOptions = [];

    /** @var Content */
    public $item = null;

    /** @var bool */
    public $last = false;

    /** @var Content */
    public $newsletter;

    /** @var string */
    public $locale;

    /**
     * @param Content $item
     * @param array $itemOptions
     * @param bool $last
     */
    public function __construct(Content $item, array $itemOptions = [], bool $last = false)
    {
        $this->itemOptions = $itemOptions;
        $this->item = $item;
        $this->last = $last;
        $this->newsletter = View::shared('newsletter');
        $this->locale = View::shared('locale');
    }

    /**
     *
     * @return array|\Closure|\Illuminate\Contracts\Support\Htmlable|\Illuminate\Contracts\View\View|\Illuminate\View\View|\Laravel\Lumen\Application|string
     */
    public function render()
    {
        return view('iln::components.partials.documents', [
            'url' => $this->getDocumentUrl(
                gedSlug: $this->itemOptions['link']['sectionLinkUrl'] ?? 'ged',
                document: $this->item,
                parentFieldName: $this->itemOptions['parentFieldName'] ?? 'folders',
                modalOptions: $this->itemOptions['modal'] ?? [],
            ),
            'logoUrl' => $this->getDocumentLogoUrl($this->item),
            'titleColor' => $this->itemOptions['titleColor'] ?? '#2C3E50',
            'dateFormat' => $this->itemOptions['dateFormat'] ?? '%d/%m/%Y',
            'dateFieldName' => $this->itemOptions['dateFieldName'] ?? 'published_at',
            'dateColor' => $this->itemOptions['dateColor'] ?? '#A7A7A7',
            'leftLineColor' => Setting::where('key', 'color_primary')->first()?->value ?? $this->itemOptions['leftLineColor'] ?? '#2C3E50',
        ]);
    }

    /**
     *
     * @param Content $document
     *
     * @return string
     */
    public function getDocumentLogoUrl(Content $document): string
    {
        $type = class_to_type($document);
        $fileType = 'other';
        // Déterminer l'extension
        if (
            Str::endsWith($type, '_link')
            && InsideSchema::hasField($type, 'link')
            && $document->link !== null
        ) {
            $fileType = 'link';
        } else {
            if (
                InsideSchema::hasField($type, 'file')
                && $document->file !== null
            ) {
                $disk = Storage::disk('local');
                // Récupere l'extension
                if ($disk->exists($document->file)) {
                    $realPath = $disk->path($document->file);
                    MimeTypeGuesser::reset();
                    $guesser = MimeTypeGuesser::getInstance();
                    $mime = $guesser->guess($realPath);
                    switch ($mime) {
                        case 'application/vnd.ms-excel':
                        case 'application/msexcel':
                        case 'application/x-msexcel':
                        case 'zz-application/zz-winassoc-xls':
                            $fileType = 'excel';
                            break;
                        case 'image/jpeg':
                        case 'image/pjpeg':
                        case 'image/png':
                        case 'image/tiff':
                        case 'image/gif':
                        case 'image/bmp':
                        case 'image/x-bmp':
                        case 'image/x-ms-bmp':
                            $fileType = 'image';
                            break;
                        case 'application/pdf':
                        case 'application/acrobat':
                        case 'application/nappdf':
                        case 'application/x-pdf':
                        case 'image/pdf':
                            $fileType = 'pdf';
                            break;
                        case 'application/vnd.ms-powerpoint':
                        case 'application/mspowerpoint':
                        case 'application/powerpoint':
                        case 'application/x-mspowerpoint':
                        case 'application/vnd.openxmlformats-officedocument.presentationml.presentation':
                            $fileType = 'powerpoint';
                            break;
                        case 'text/plain':
                            $fileType = 'text';
                            break;
                        case 'video/mp4':
                        case 'video/mp4v-es':
                        case 'video/x-m4v':
                        case 'video/x-matroska':
                        case 'video/quicktime':
                        case 'video/avi':
                        case 'video/divx':
                        case 'video/msvideo':
                        case 'video/vnd.divx':
                        case 'video/x-avi':
                        case 'video/x-msvideo':
                            $fileType = 'video';
                            break;
                        case 'application/msword':
                        case 'application/vnd.ms-word':
                        case 'application/x-msword':
                        case 'zz-application/zz-winassoc-doc':
                            $fileType = 'word';
                            break;
                        case 'application/zip':
                        case 'application/x-zip':
                        case 'application/x-zip-compressed':
                            $fileType = 'zip';
                            break;
                    }
                }
            }
        }

        return url(
            '/vendor/maecia/inside-newsletters/resources/images/documents/'
            .$fileType.'.jpg'
        );
    }

    /**
     * @param string $gedSlug
     * @param Content $document
     * @param string|null $parentFieldName
     *
     * @return string
     */
    public function getDocumentUrl(
        string $gedSlug,
        Content $document,
        ?string $parentFieldName = null,
        array $modalOptions = [],
    ): string {
        $type = class_to_type($document);

        // C'est un document lien
        if (
            Str::endsWith($type, '_link')
            && InsideSchema::hasField($type, 'link')
        ) {
            return $document->link ?? '#';
        }

        // On vérifie si c'est slugable
        if ($document->isAliasable) {
            return url('/'.Arr::first($document->slug));
        }

        $parentFieldName = $parentFieldName ?? 'folders';
        if (! InsideSchema::hasField($type, $parentFieldName)) {
            Log::error(
                '[Documents::getDocumentUrl] wrong newsletter config type ('
                .$type.') does not have a parent ('.$parentFieldName.')'
            );

            return '#';
        }

        $parent = $document->{Str::camel($parentFieldName)};
        if ($parent !== null) {
            $parent = $parent->first();
        }
        if ($parent === null) {
            return '#';
        }

        if (Package::has('inside-dms')) {
            $modalName = $modalOptions['name'] ?? 'doc-ged';
            $modalSize = $modalOptions['size'] ?? 'm';
            $modalCloseAll = $modalOptions['closeAll'] ?? true;

            $query = http_build_query([
                'folder' => $parent->uuid,
                'size' => $modalSize,
                'modal' => $modalName,
                'contentType' => $type,
                'uuid' => $document->uuid,
                'closeAll' => $modalCloseAll,
            ]);

            return url("/$gedSlug?$query");
        }

        return url('/'.$gedSlug.'#&ged/'.$parent->uuid.'&modal/'.$type.'/'.$document->uuid);
    }
}
