<?php

namespace Inside\Newsletters\Views\Components;

use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\View;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Newsletters;
use Inside\Content\Models\Contents\Users;
use Inside\Kernel\Filesystem\Facades\FileStreamer;
use Inside\Newsletters\Services\NewsletterConfigService;
use Inside\Support\View\Component;
use Symfony\Component\Yaml\Yaml as SymfonyYaml;
use Inside\Newsletters\Facades\UrlSigner;

class Layout extends Component
{
    /** @var Content */
    public $newsletter;

    /** @var Collection */
    public $options;

    public function __construct(Content $newsletter, array $data, string $name = null)
    {
        $this->newsletter = $newsletter;
        $user             = Users::where('email', config('app.technical_mail'))->first();
        if (isset($data['to.uuid'])) {
            $user = Users::find($data['to.uuid']);
        }
        $this->options = collect($data);

        if ($name !== null) {
            $this->options = $this->options->merge($this->getNewsletterConfig($name));
        }

        $newsletterSettingsConfig = NewsletterConfigService::loadFromSettings(
            $this->options['_meta']['type'],
            $user->langcode ?? config('app.locale')
        );

        $optionsArray = $this->options->toArray();

        $newsletterSettingsConfig->each(function ($value, $key) use (&$optionsArray) {
            Arr::set($optionsArray, $key, $value);
        });

        $this->options = collect($optionsArray);

        $this->overloadSettingsByNewsletterFields($newsletter);

        $fieldsPrefix = $this->options['_meta']['fieldsPrefix'] ?? ('iln'.($name[0] ?? '').'_');

        $locale = $user->langcode;
        App::setLocale($locale);

        $trackingPixelUrl = UrlSigner::signedRoute('newsletters.viewed', ['type' => $newsletter->content_type, 'newsletterUuid' => $newsletter->uuid, 'userUuid' => $user->uuid]);

        View::share('newsletter', $this->newsletter);
        View::share('options', $this->options);
        View::share('user', $user);
        View::share('locale', $locale);
        View::share('fieldsPrefix', $fieldsPrefix);
        View::share('trackingPixelUrl', $trackingPixelUrl);
    }

    public function render()
    {
        return view('iln::layout', ['options' => $this->options]);
    }

    protected function getNewsletterConfig(string $name): Collection
    {
        $fileName = back_path('resources/newsletters/' . $name . '.php');
        if (!File::exists($fileName)) {
            Log::error('Newsletter config file [' . $fileName . '] does not exists');

            return collect();
        }
        return collect(require $fileName);
    }

    private function overloadSettingsByNewsletterFields(Content $newsletter): void
    {
        $optionsArray = $this->options->toArray();
        $storage = Storage::disk('local');
        collect([
            'nl_banner' => 'banner.backgroundImageUrl',
            'nl_logo' => 'logo.logoUrl',
        ])->each(function ($optionKey, $newsletterField) use (
            $newsletter,
            $storage,
            &$optionsArray
        ) {
            if (
                !empty($newsletter->getAttribute($newsletterField)) &&
                $storage->exists($newsletter->getAttribute($newsletterField))
            ) {
                Arr::set(
                    $optionsArray,
                    $optionKey,
                    $this->newsletter->uuid
                );
            }
        });
        $this->options = collect($optionsArray);
    }
}
