<?php

namespace Inside\Newsletters\Services;

use Illuminate\Console\OutputStyle;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Newsletters;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Newsletters\Exceptions\NewsletterConfigNotFound;
use Inside\Settings\Models\Setting;
use Inside\Content\Facades\Schema;

class NewsletterSettingService
{
    private const GROUP = 'newsletter';
    private const LOGO_KEY = 'logo.logoUrl';
    private const BANNER_KEY = 'banner.backgroundImageUrl';

    private OutputStyle $output;
    private array $config = [];


    /**
     * @param OutputStyle $output
     * @return $this
     */
    public function attachCmdOutput(OutputStyle $output): self
    {
        $this->output = $output;
        return $this;
    }

    /**
     * @return void
     */
    public function execute(): void
    {
        $newsletters = collect(config('newsletters.types'))->pluck('notification.view');
        if ($newsletters->count() !== 1) {
            $this->output->warning('Newsletter settings : not available when multiple newsletter types exists');
            return;
        }
        try {
            $this->loadConfig()
                ->createOrUpdateSetting();
        } catch (NewsletterConfigNotFound $exception) {
            $this->output->warning($exception->getMessage());
            Log::warning($exception->getMessage());
        }
    }

    private function loadConfig(): self
    {
        $newsletter = config('app.code');
        $configPath = back_path("resources/newsletters/".$newsletter.".php");
        if (!file_exists($configPath)) {
            throw new NewsletterConfigNotFound("Configuration file for newsletter {$newsletter} does not exist.");
        }

        $this->config = include $configPath;

        return $this;
    }

    private function createOrUpdateSetting(): self
    {
        collect([
            'newsletters_logo' => $this->getFile(self::LOGO_KEY),
            'newsletters_banner' => $this->getFile(self::BANNER_KEY),
            'newsletters_display_hello' => true,
            'show_newsletter_button' => true
        ])->each(function ($value, $key) {
            Setting::firstOrCreate(
                ['group' => self::GROUP, 'key' => $key],
                ['value' => $value]
            );
        });

        collect([
            'name' => null,
            'description' => null,
            'text_color' => '#fff',
        ])->each(function ($value, $key) {
            $this->saveSettingWithLang($key, $value);
        });

        return $this;
    }

    /**
     * @param string $fileKey
     * @return string|null
     */
    private function getFile(string $fileKey): ?string
    {
        $filePath = Arr::get($this->config, $fileKey);
        if (is_array($filePath)) {
            $filePath = $filePath[config('app.locale')]??$filePath['fr'];
        }

        if (!file_exists($filePath) || !is_file($filePath)) {
            return null;
        }

        return $this->storeFile($filePath);
    }

    /**
     * @param string $filePath
     * @return string
     * @throws \Illuminate\Contracts\Filesystem\FileNotFoundException
     */
    private function storeFile(string $filePath): string
    {
        $extension = str($filePath)->afterLast('.')->start(".");
        $chunckId = Str::uuid();
        $directoryPath = "assets/system";
        $file = "$directoryPath/{$chunckId}.{$extension}";
        $fileContent = (string) file_get_contents($filePath);

        if (!Storage::disk('local')->exists($directoryPath)) {
            Storage::makeDirectory($directoryPath);
        }
        Storage::disk('local')->put($file, $fileContent);
        return $file;
    }

    /**
     * @param string $key
     * @param string|null $value
     * @return void
     */
    private function saveSettingWithLang(string $key, ?string $value): void
    {
        $newsletterType = Arr::get($this->config, '_meta.type');

        foreach (list_languages() as $language) {
            $completeKey = sprintf('%s_%s_%s', $newsletterType, $key, $language);
            Setting::firstOrCreate(
                ['group' => self::GROUP, 'key' => $completeKey],
                ['value' => $value]
            );
        }
    }
}
