<?php

namespace Inside\Newsletters\Http\Controllers;

use Illuminate\Http\Response;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Services\Queries\ContentQuery;
use Illuminate\Support\Facades\DB;
use Inside\Newsletters\Exports\NewslettersEventsExport;
use Inside\Newsletters\Exports\NewslettersStatisticsExport;
use Inside\Newsletters\Models\NewslettersSent;
use Inside\Newsletters\Services\UrlSignerService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Carbon;
use Inside\Newsletters\Services\StatisticsService;
use Illuminate\Http\JsonResponse;
use Inside\Statistics\Listeners\NotifyUserOfCompletedStatisticExport;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Pagination\Paginator;
use Inside\Newsletters\Models\NewslettersStatisticsGeneric;
use Illuminate\Database\Eloquent\Collection;

class NewslettersStatsController
{
    public const TRANSPARENT_PIXEL = 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mNkYAAAAAYAAjCB0C8AAAAASUVORK5CYII=';

    public function types(): JsonResponse
    {
        $typesConfig = config('newsletters')['types'];

        return response()->json(collect($typesConfig)->mapWithKeys(function ($typeConfig, $type) {
            return [$type => isset($typeConfig['title']) ? __($typeConfig['title']) : null];
        })->toArray());
    }

    public function list(Request $request, StatisticsService $statisticsService): Paginator|Collection|LengthAwarePaginator|array
    {
        $search = $request->query->get('query') ?: null;
        $filters = ContentHelper::extractFiltersInputFromRequest($request);
        $query = $statisticsService->getNewslettersStatisticsQuery($filters, $search);
        $count = $statisticsService->getQueryCount($query);

        return ContentHelper::getResultFromQuery($query, $filters, true, $count);
    }

    public function views(string $newsletterUuid, Request $request, StatisticsService $statisticsService): JsonResponse
    {
        $search = $request->query->get('query') ?: null;
        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        return response()->json($statisticsService->getNewsletterViews($newsletterUuid, $search, $filters));
    }

    public function viewsExport(string $newsletterUuid, Request $request): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();
        $filters = ContentHelper::extractFiltersInputFromRequest($request);
        $excelPath = 'statistics/newsletters_events_'.now('Europe/Paris')->format('Y_m_d_His').'_export.xls';
        $search = $request->query->get('query') ?: null;

        (new NewslettersEventsExport($newsletterUuid, ($user?->langcode ?? 'fr'), $filters, $search))
            ->queue($excelPath, 'protected')
            ->chain([
                new NotifyUserOfCompletedStatisticExport($user, $excelPath, 'protected'),
            ])
            ->allOnQueue(get_low_priority_queue_name());

        return response()->json(['success' => true]);
    }

    public function viewed(string $type, string $newsletterUuid, string $userUuid, Request $request, UrlSignerService $urlSigner): Response
    {
        if (! $urlSigner->hasValidSignature($request)) {
            Log::error('Invalid signature for newsletter (viewed)', [
                'newsletter_uuid' => $newsletterUuid,
                'user_uuid' => $userUuid,
                'signature' => $request->get('signature'),
                'app_key' => config('app.key'),
            ]);
        }

        if (NewslettersSent::query()->where('uuid', $newsletterUuid)->where('user_uuid', $userUuid)->exists()) {
            $statsTable = type_to_stats_table($type);

            if ($statsTable === null) {
                throw new \Exception('Provided type does not exists !');
            }

            DB::table($statsTable)->insert([
                'statistic_type' => NewslettersStatisticsGeneric::TYPE_VIEWED,
                'content_uuid' => $newsletterUuid,
                'user_uuid' => $userUuid,
            ]);
        }

        return response(base64_decode(self::TRANSPARENT_PIXEL))
            ->header('Content-Type', 'image/png');
    }

    public function clicked(string $type, string $newsletterUuid, string $userUuid, Request $request, UrlSignerService $urlSigner): RedirectResponse
    {
        if (! $urlSigner->hasValidSignature($request)) {
            Log::error('Invalid signature for newsletter (clicked)', [
                'newsletter_uuid' => $newsletterUuid,
                'user_uuid' => $userUuid,
                'signature' => $request->get('signature'),
                'app_key' => config('app.key'),
            ]);
        }

        if (NewslettersSent::query()->where('uuid', $newsletterUuid)->where('user_uuid', $userUuid)->exists()) {
            $statsTable = type_to_stats_table($type);

            if ($statsTable === null) {
                throw new \Exception('Provided type does not exists !');
            }

            DB::table($statsTable)->insert([
                'statistic_type' => NewslettersStatisticsGeneric::TYPE_CLICKED,
                'content_uuid' => $newsletterUuid,
                'user_uuid' => $userUuid,
            ]);
        }

        $redirect = $request->query('redirect');

        if (! is_string($redirect)) {
            $redirect = '';
        }

        return redirect("/$redirect");
    }

    public function export(Request $request): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();
        $filters = ContentHelper::extractFiltersInputFromRequest($request);
        $excelPath = 'statistics/newsletters_'.now('Europe/Paris')->format('Y_m_d_His').'_export.xls';

        (new NewslettersStatisticsExport($user?->langcode ?? 'fr', $filters))
            ->queue($excelPath, 'protected')
            ->chain([
                new NotifyUserOfCompletedStatisticExport($user, $excelPath, 'protected'),
            ])
            ->allOnQueue(get_low_priority_queue_name());

        return response()->json(['success' => true]);
    }
}
